import java.awt.geom.GeneralPath;

/**
 * This class stores the information necessary to run the
 * search algorithm for periodic billiard paths in triangles.
 */
public class SlalomHull{
    // left side and right side of the hull
    // both of these will be lists containing Complex numbers
    PatList ls,rs;
    
    public class DataBackup {
        public Complex [] left,right;
        public DataBackup(Complex []l, Complex []r){
            left=l;
            right=r;
        }
    }
    
    
    /** Constuct a SlalomHull with one point on the left and one
     * on the right.
     */
    public SlalomHull(Complex left,
    Complex right){
        ls=new PatList(150);
        ls.add(left);
        rs=new PatList(150);
        rs.add(right);
    }
    
    public DataBackup backup(){
        return new DataBackup(ls.extractData(), rs.extractData());
    }
    
    public void recover(DataBackup db) {
        ls.replaceData(db.left);
        rs.replaceData(db.right);
    }
    
    /** return the number of vertices in the hull */
    public int size(){
        return ls.size+rs.size;
    }
    
    public int leftSize(){
        return ls.size;
    }
    
    public int rightSize(){
        return ls.size;
    }
    
    public Complex leftGet(int i){
        return ls.get(i);
    }
    
    public Complex rightGet(int i){
        return rs.get(i);
    }
    
    /** remove the last left element from the hull */
    private void removeLastLeft() {
        ls.removeLast();
    }
    
    /** remove the right left element from the hull */
    private void removeLastRight() {
        rs.removeLast();
    }
    
    /** remove the first left element from the hull */
    private void removeFirstLeft() {
        ls.removeFirst();
    }
    
    /** remove the right left element from the hull */
    private void removeFirstRight() {
        rs.removeFirst();
    }
    
    /**
     * Get a copy of the left side of the hull.
     * <br><br>
     * This is only for demonstration purposes... presumably it is quite slow.
     * @return Returns a PatList of Complex numbers
     *
     * public PatList getLeftList(){
     * return new PatList(ls);
     * }
     */
    
    /**
     * Get a copy of the right side of the hull.
     * <br><br>
     * This is only for demonstration purposes... presumably it is quite slow.
     * @return Returns a PatList of Complex numbers
     *
     * public PatList getRightList(){
     * return new PatList(rs);
     * }
     */
    
    
    /** Return false if adding z to the left will result in the hull being destroyed. Otherwise
     * return true;
     */
    public boolean survivesLeftAdd(Complex z){
        return !(TriangleGeometry.sign(rs.getLast(),ls.getFirst(),z)==1);
    }
    
    /** Return false if adding z to the right will result in the hull being destroyed. Otherwise
     * return true;
     */
    public boolean survivesRightAdd(Complex z){
        return !(TriangleGeometry.sign(ls.getLast(),rs.getFirst(),z)==-1);
    }
    
    /** Add a point to the left side of the hull.
     * <br><br>
     * If the function returns false, adding further points
     * to the hull will result in errors.
     *
     * @return true if succeeds, false if the hull is destroyed
     * (that is, there is no line which stays to left of the points
     * in the left side of the hull and right of the ones on the right)
     */
    public boolean addToLeft(Complex z) {
        //System.out.println("rs_first x="+rs.getFirst().x+" y="+rs.getFirst().y);
        //System.out.println("ls_last x="+ls.getLast().x+" y="+ls.getLast().y);
        
        if (TriangleGeometry.sign(rs.getFirst(),z,ls.getLast())==1) {
            // the point belongs in the hull
            
            // first try to remove vertices from the hull on the right side
            while ((rs.size>1)&&(TriangleGeometry.sign(rs.getFirst(),z,
            rs.getSecond())==1))
                // permenantly remove the first on the right side
                rs.removeFirst();
            
            if ((rs.size==1)&&(TriangleGeometry.sign(ls.getFirst(),z,
            rs.getFirst())==1))
                // rs.getFirst() should also be removed
                // now the right side contains no elements.
                // thus the hull has degenerated
                return false;
            
            
            // now look at the left
            while ((ls.size>1)&&(TriangleGeometry.sign(ls.getSecondToLast(),z,
            ls.getLast())==1))
                // remove the last point from the left side permenently, and move on.
                ls.removeLast();
            
        if ((ls.size==1)&&(TriangleGeometry.sign(z,rs.getLast(),ls.getFirst())==1)) {
                // we should remo0ve the last remaining point on the left
                // this means that the only remaining point on the left is z itself
                // Actually- this should never happen
                return false;
            }
            
            // finally add z to the left side.
            ls.add(z);
            return true; // the hull is fine
        }
        // no modification needed
        // the point does not belong in the hull
        return true;
    }
    
    /** Add a point to the right side of the hull.
     * <br><br>
     * If the function returns false, adding further points
     * to the hull will result in errors.
     *
     * @return true if succeeds, false if the hull is destroyed
     * (that is, there is no line which stays to left of the points
     * in the left side of the hull and right of the ones on the right)
     */
    public boolean addToRight(Complex z) {
        if (TriangleGeometry.sign(ls.getFirst(),z,rs.getLast())==-1) {
            // the point belongs in the hull
            
            // first try to remove vertices from the hull on the right side
            while ((ls.size>1)&&(TriangleGeometry.sign(ls.getFirst(),z,
            ls.getSecond())==-1))
                // permenantly remove the first on the right side
                ls.removeFirst();
            
            if ((ls.size==1)&&(TriangleGeometry.sign(rs.getFirst(),z,
            ls.getFirst())==-1))
                // ls.getFirst() should also be removed
                // now the right side contains no elements.
                // thus the hull has degenerated
                return false;
            
            
            // now look at the left
            while ((rs.size>1)&&(TriangleGeometry.sign(rs.getSecondToLast(),z,
            rs.getLast())==-1))
                // remove the last point from the left side permenently, and move on.
                rs.removeLast();
            
            if ((rs.size==1)&&(TriangleGeometry.sign(z,ls.getLast(),rs.getFirst())==-1)) {
                // we should remo0ve the last remaining point on the left
                // this means that the only remaining point on the left is z itself
                // Actually- this should never happen
                return false;
            }
            
            // finally add z to the left side.
            rs.add(z);
            return true; // the hull is fine
        }
        // no modification needed
        // the point does not belong in the hull
        return true;
    }
    
    /** z is a vector pointing in the direction of holonomy of the unfolding */
    public boolean strongTest(Complex z){
        return (
        (z.times(rs.getLast().minus(ls.getFirst()).conjugate()).y>0) &&
        (z.times(ls.getLast().minus(rs.getFirst()).conjugate()).y<0));
    }
}