
/*FANCY DRAWING ROUTINES*/



/*all the routines depend on the choice of
a center for a tile.  If the chosen center
lies too close to the edge, there is some
chance that the plotting won't work.
This first routine attempts to change
the center so that it is more
in the middle of the tile.*/


complex new_center(w,z)
     word w;
     complex z;
{
  int i;
  complex step;
  complex zz[16];
  double ii;
  zz[0]=convert(0.0,0.0);
  for(i=1;i<=12;++i) {
    ii=2.0*Pi*i/12.0;
    step=convert(cos(ii),sin(ii));
    zz[i]=find_edge(w,z,step,.000000000001);
    zz[0]=plus(zz[0],zz[i]);
  }
  zz[0].x=zz[0].x/12.0;
  zz[0].y=zz[0].y/12.0;
  return(zz[0]);
}



int right_test(z)
     complex z;
{
  double d;
  d=z.x+z.y;
  if(fabs(d-1.0)<.00000001) return(1);
  return(0);
}


/*Newton's method apparatus - first 4 routines*/

/*this routine returns the value of the
defining function for the word w at the
complex number zz. */


double edge_fcn(U,zz,k) 
     unfold U;
     complex zz;   //approximate location
     int k;
{
    double test1;
    triangle_list X;
    complex z1,z2,z3,z4,z5,z6;
    int a,b;
    unfold U1;

    a=zz.a[1];
    b=zz.a[2];
    if(k==2) {
    a=zz.a[3];
    b=zz.a[4];
    }

    U1=assign_geometry(U,zz);
    z1=top_position(U1,0);
    z2=top_position(U1,U1.w[1].l);
    z3=top_position(U1,a);
    z4=bottom_position(U1,b);
    test1=area(z1,z2,z3)-area(z1,z2,z4);
    return(test1);
}








/*this routine computes the approximate gradient
of the defining function.  The value .00001
could of course be replaced by some other
small value. Since the gradients are nearly
constant in the scale of interest, the
value doesn't seem that important.*/


complex gradient(U,zz,k) 
     unfold U;
     complex zz;   //approximate location
     int k;
{
    complex zzx,zzy;
    complex grad;
    unfold U1,U2;
    zzx=zz;
    zzy=zz;
    zzx.x=zz.x+.000001;
    zzy.y=zz.y+.000001;
    U1=assign_geometry(U,zz);
    U2=assign_geometry(U,zzx);
    grad.x=1000000.0*(edge_fcn(U2,zzx,k)-edge_fcn(U1,zz,k));
    U2=assign_geometry(U,zzy);
    grad.y=1000000.0*(edge_fcn(U2,zzy,k)-edge_fcn(U1,zz,k));
    return(grad);
}







/*This routine starts with a complex number which
approximately lies on a vertex, then finds the
exact vertex using Newton's method. 
Technically, we compute the gradients once,
and don't update during the iteration. 
This saves a lot of time.   
While this isn't precisely newton's method,
it works well because the gradient is usually
nearly constant in the region of interest.
The method deals specially with points near
(0,1), (1,0), (1,1) because newton's method
breaks down in this range.*/





complex newton_vertex(U,zz,debug)
     unfold U;
     complex zz;   //approximate location
     int debug;
{
  double test[5];
  complex grad[3];
  matrix2 m1,m2;
  complex y,yy;
  unfold U1;
  int j;
  
  yy=zz;  
  U1=assign_geometry(U,yy);
  grad[1]=gradient(U1,yy,1);
  grad[2]=gradient(U1,yy,2);


  for(j=1;j<=5;++j) {
  U1=assign_geometry(U,yy);
  test[1]=edge_fcn(U1,yy,1);
  test[2]=edge_fcn(U1,yy,2);

  m1.a[1][1]=grad[1].x;
  m1.a[2][1]=grad[2].x;
  m1.a[1][2]=grad[1].y;
  m1.a[2][2]=grad[2].y;
  m2=inverse_matrix(m1);
  y.x=m2.a[1][1]*test[1]+m2.a[1][2]*test[2];
  y.y=m2.a[2][1]*test[1]+m2.a[2][2]*test[2];
  yy=minus(yy,y);
  yy.a[1]=zz.a[1];
  yy.a[2]=zz.a[2];
  yy.a[3]=zz.a[3];
  yy.a[4]=zz.a[4];

  if(debug==1) {
    printf("debugging newton_vertex\n");
    dpt(test[1]);
    dpt(test[2]);
    cpt(grad[1]);
    cpt(grad[2]);
    cpt3(yy);
    cpt(y);
    printf("\n\n");
  }
  }  
  return(yy);
}










  /*This routine starts with a point near an
  edge of the tile and pushes the point
  onto the edge using Newton's method.
  The same remarks apply here, as above,
  concerning updating the gradient function. */


complex newton_edge(U,z,debug)
     unfold U;
     complex z;
     int debug;
{
  int i;
  double test,old_test;
  complex grad;
  complex yy,y;

  yy=z;
   grad=gradient(U,yy,1);
   test=1.0;
   i=0;
   while((i<10)&&(fabs(test)>.0000000001)) {
     ++i;
     test=edge_fcn(U,yy,1);
     /*in case the method seems to be hung up
       we recompute the gradient.*/
     if(fabs(test)>fabs(old_test)) {
        grad=gradient(U,yy,1);
     }

    y.x=grad.x*test/(norm(grad)*norm(grad));
    y.y=grad.y*test/(norm(grad)*norm(grad));
    yy=minus(yy,y);
    yy.a[1]=z.a[1];
    yy.a[2]=z.a[2];
    old_test=test;
  }
  return(yy);
}

/*that's the end of the Newton's method
  apparatus.*/




/*routines for locating the vertices.*/


/*this routine figures out which defining
functions at a vertex really come from
an edge.  There is a small tolerance
which needs to be specified */






complex get_edge_leaders(U,z,v,k) 
     unfold U;
     complex z; //center point
     complex v; //point of interest
     int k;     //clockwise versus counterclockwise
{
  word w1,w2;
  int i,j;
  complex grad;
  complex y,z1;
  double min,test;
  unfold V;

  y=v;
  V=assign_geometry(U,v);

  //printf("\n\n\n get edge leaders debugging\n\n");
  w1=get_all_leaders(V,1);
  w2=get_all_leaders(V,2);
  min=10000000.0;

  for(i=1;i<=w1.l;++i) {
    for(j=1;j<=w2.l;++j) {
      y.a[1]=w1.n[i];
      y.a[2]=w2.n[j];
      grad=gradient(V,y,1);
      grad=unit(grad);
      z1=plus(y,times(convert(0.0,dist(z,v)),grad));
      if(k*area(v,z,z1)<0.0)
      z1=minus(y,times(convert(0.0,dist(z,v)),grad));
      test=dist(z,z1);   
      //dpt(test);
      if(min>test) {
	min=test;
	y.a[3]=y.a[1];
	y.a[4]=y.a[2];
      }
    }}
  y.a[1]=y.a[3];
  y.a[2]=y.a[4]; 
  //printf("\n\n\n end get edge leaders debugging\n\n");
  return(y);
}








/*Here are the main routines.  
It was extremely difficult
to get these to work.*/


/*the first three routines test
whether or not a point lies on
the edge of a tile.  The last routine
uses the bisection method along an
edge to find the next vertex.
Repeated calls to newton's method
keep the point of interest on the
edge.  The edge testers are
invoked to decide if a point has
reached the end of the edge.
The edge testers are not entirely
reliable.  I have several different
routines to help make the overall
routine more reliable.
*/



/*this test is defined as follows: We
retract our point slightly towards
the center of the tile and then
just check if it is inside the tile*/



int edge_test(U,z,qq,debug)
     unfold U;
     complex z,qq;
     int debug;
{
  complex q[5];
  double scale;
  unfold V;
  double e;
  int test;
  q[0]=qq;
  scale=dist(z,qq);
  e=.05*scale;
  q[1].x=e*z.x+(1.0-e)*q[0].x;
  q[1].y=e*z.y+(1.0-e)*q[0].y;
  V=assign_geometry(U,q[1]);
  test=corridor_test(V);
  return(test);
}


complex get_next_vertex(U,z,v,debug)
     unfold U;
     complex z,v;
     int debug;
{

  complex q[5];
  triangle_list J;
  int test;
  int i,j;
  int count;
  double e;
  complex grad,step;
  word W;

  /*gets the direction of the edge*/

  grad=unit(gradient(U,v,1));
  step=times(grad,convert(0.0,1.0));


  e=0.000000001;
  q[1]=v;
  q[3]=extrapolate(v,step);

  if(debug==1) {
    cpt(step);
    cpt(q[3]);
  }

  /*re-do extrapolation for points near edge*/
  if((q[3].x>.99) && (step.y>.99)) q[3]=convert(1.0,1.0);
  if((q[3].y>.99) && (step.x>.99)) q[3]=convert(1.0,1.0);
  if((q[3].x>.99) && (step.y<-.99)) q[3]=convert(1.0,0.0);
  if((q[3].y>.99) && (step.x<-.99)) q[3]=convert(0.0,1.0);

  /*recognize corners*/
  if(q[3].x<.05) q[3].x=0.0;
  if(q[3].x>.95) q[3].x=1.0;
  if(q[3].y<.05) q[3].y=0.0;
  if(q[3].y>.95) q[3].y=1.0;


  if(debug==1) {
    printf("the distant point:\n");
    cpt(q[3]);
  }

  count=0;
  while(dist(q[1],q[3])>e)
    {  
      ++count;
      if(count>100)  {
	printf("count too high\n");
             exit(1);
      }

      /*for the first few counts we 
        move out more conservatively*/

      q[2]=interpolate(q[1],q[3]);
      if(count<4)  q[2]=interpolate(q[1],q[2]);
      if(count<2)  q[2]=interpolate(q[1],q[2]);


      q[2].a[1]=v.a[1];
      q[2].a[2]=v.a[2];
      q[2]=newton_edge(U,q[2],0);

      test=edge_test(U,z,q[2],debug);
      if(test==0) q[3]=q[2];
      if(test==1) q[1]=q[2];

      /***********************************/
      /*recognize points near the corners*/

      if((q[2].x>.99) && (q[2].y<.05)) {
	q[2].a[1]=-1;
	return(q[2]);
      }


      if((q[2].y>.99) &&(q[2].x<.05)) {
	q[2].a[1]=-2;
	return(q[2]);
      }


      if((q[2].x>.99) &&(q[2].y>.99)) {
	q[2].a[1]=-3;
	return(q[2]);
      }
      /***********************************/


      /*note the massive debugging*/
      if((count<20)&&(debug==1)) {
	printf("\n\n");	
        printf("count %d\n\n",count);
        printf("test: %d\n",test);	
	printf("center:"); cpt(z);
        printf("vertex:"); cpt3(v);
	W=get_all_leaders(assign_geometry(U,v),1);
	printf("vertex leaders: \n");
	wpt2(W);	
	W=get_all_leaders(assign_geometry(U,v),2);
	printf("\n");
	wpt2(W);
	printf("\n");
	printf("gradient: ");
	cpt(grad);
	printf("edge direction: ");
	cpt(step);
	test=corridor_test(U);
        printf("inside tile (1 means yes):%d\n",test);
	W=get_all_leaders(assign_geometry(U,q[2]),v,1);
	printf("point2 leaders: \n");
	wpt2(W);	
        W=get_all_leaders(assign_geometry(U,q[2]),v,2,0);
	printf("\n");
	wpt2(W);
	printf("\n");
	printf("inside tile:%d\n",test);
	printf("the triple:\n");
	printf("point 1:"); cpt3(q[1]);
	printf("point 2:"); cpt3(q[2]);
	printf("point 3:"); cpt3(q[3]);  
	printf("point 2 leaders: \n");
	wpt2(W);
	printf("\n");	
        W=get_all_leaders(assign_geometry(U,q[2]),q[2],2);
	wpt2(W);
	printf("\n");
      }

    }

  if(debug==1) 
    {
	printf("final point 3 leaders:\n"); 
	W=get_all_leaders(assign_geometry(U,q[2]),q[3],1);
	wpt2(W);
	printf("\n");	
        W=get_all_leaders(assign_geometry(U,q[2]),q[3],2);
	wpt2(W);	
        printf("\n");
    }


  q[3]=get_edge_leaders(U,z,q[3],1);  
  q[2].a[1]=v.a[3];
  q[2].a[2]=v.a[4];
  q[2].a[3]=q[3].a[1];
  q[2].a[4]=q[3].a[2];
  q[2]=newton_vertex(U,q[2],0);
  return(q[2]);
}

















/*special routines for vertices near the corners.
  The basic idea here is to replace the tile
  edge by a segment which is either

     vertical
     horizontal
     diagonal

depending on which corner we are near.
This edge runs through the tile.
We then force the point to move
along this segment until it hits
another edge.  Effectively, we are
skirting around (but nearby to) the
supposed corner vertex. */


complex get_next_vertex1(U,z,v,debug) 
     unfold U;
     complex z,v;
     int debug;
{
  complex q[5];
  double d1,d2,d3,test;
  complex grad;
  unfold U1;
  q[1]=v;
  grad=convert(1.0,0.0);

  d1=0.0;
  d2=0.1;
  d3=1.0;

  while(fabs(d3-d1)>.0000000001) {
    d2=.9*d1+.1*d3;
    q[2]=plus(q[1],times(grad,convert(d2,0.0)));
    U1=assign_geometry(U,q[2]);
    test=corridor_test(U1);
    if(test==1) d1=d2;
    if(test==0) d3=d2;
  }


  if(d2<.000000001) {
  grad=convert(-1.0,0.0);
  d1=0.0;
  d2=0.1;
  d3=1.0;

  while(fabs(d3-d1)>.0000000001) {
    d2=.9*d1+.1*d3;
    q[2]=plus(q[1],times(grad,convert(d2,0.0)));
    U1=assign_geometry(U,q[2]);
    test=corridor_test(U1);
    if(test==1) d1=d2;
    if(test==0) d3=d2;
  }
  }

  q[2].a[1]=-2;
  return(q[2]);
}




complex get_next_vertex2(U,z,v,debug) 
     unfold U;
     complex z,v;
     int debug;
{
  complex q[5];
  double d1,d2,d3,test;
  complex grad; 
  unfold U1;
  q[1]=v; 
  grad=convert(0.0,-1.0);

  d1=0.0;
  d2=0.1;
  d3=1.0;

  while(fabs(d3-d1)>.0000000001) {
    d2=.9*d1+.1*d3;
    q[2]=plus(q[1],times(grad,convert(d2,0.0)));  
    U1=assign_geometry(U,q[2]);
    test=corridor_test(U1);
    if(test==1) d1=d2;
    if(test==0) d3=d2;
  }

  if(d2<.000000001) {

 grad=convert(0.0,1.0);

  d1=0.0;
  d2=0.1;
  d3=1.0;

  while(fabs(d3-d1)>.0000000001) {
    d2=.9*d1+.1*d3;
    q[2]=plus(q[1],times(grad,convert(d2,0.0)));
    U1=assign_geometry(U,q[2]);
    test=corridor_test(U1);
    if(test==1) d1=d2;
    if(test==0) d3=d2;
  }
  }

  q[2].a[1]=-2;
  return(q[2]);
}



complex get_next_vertex3(U,z,v,debug) 
     unfold U;
     complex z,v;
     int debug;
{
  complex q[5];
  double d1,d2,d3,test;
  complex grad;
  unfold U1;

  q[1]=v;
  grad=convert(-1.0,1.0);

  d1=0.0;
  d2=0.1;
  d3=1.0;

  while(fabs(d3-d1)>.0000000001) {
    d2=.9*d1+.1*d3;
    q[2]=plus(q[1],times(grad,convert(d2,0.0)));  
    U1=assign_geometry(U,q[2]);
    test=corridor_test(U1);
    if(test==1) d1=d2;
    if(test==0) d3=d2;
  }

  q[2].a[1]=-3;
  return(q[2]);
}














/*computes the center of mass of a list of vertices.
  Essentially this is used to find a point well
  inside the tile.  It seems that the center of
  mass of the vertices always lies in the tile,
  though I don't have a proof.*/


complex center_of_mass(V)
     vertex_list V;
{
  complex z;
  int i;
  z=convert(0.0,0.0);
  for(i=1;i<=V.l;++i) {
    z=plus(z,V.v[i]);
  }
  z.x=z.x/V.l;
  z.y=z.y/V.l;
  return(z);
}




/*Here is the next most important routine.
  This routine gets the list of vertices.
  It starts with a seed point on an edge
  of a tile, then repeatedly invokes the
  get_next_vertex routine until the first
  and last vertices match.  For tiles which
  seem to have a vertex at the corner
  points (1,0) and (0,1) we make a cutoff
  and then, at the end, just move the
  vertices to these corners.  This will
  not cause a problem in practice. However,
  if the user deliberately searches near
  a corner, the whole thing falls apart.
  Anyway, we don't care about things
  near these corners because nig neighborhoods
  of the corner are known by us to be
  covered by tiles.*/



vertex_list get_vertex_list(U,w,z) 
     complex z;
     word w;
     unfold(U);
{

  vertex_list V;
  complex seed;
  int i,debug,match,corner1,corner2,corner3;
  double DIST,degenerate;
  complex edge,step,cm;
  FILE *fopen(),*temp;

  debug=0;   
  match=0;   
  i=1; 
  corner1=0;
  corner2=0;
  corner3=0;

  seed=find_edge(w,z,convert(-1.0,-1,0),.00000000001);
  U=edge_structure(w);
  seed=get_edge_leaders(U,z,seed,0);

  printf("seed: "); cpt3(seed);

  V.l=0;
  V.v[0]=get_next_vertex(U,z,seed,0); 

  if(V.v[0].a[1]>0)  V.v[0]=get_edge_leaders(U,z,V.v[0],0);

  printf("vertex 0: ");
  cpt3(V.v[0]);

   while(match==0) {

     if(V.v[i-1].a[1]>0) {
       V.v[i]=get_next_vertex(U,z,V.v[i-1],debug);
       if(V.v[i].a[1]>0)  V.v[i]=get_edge_leaders(U,z,V.v[i],1);
       if(corner1==1) corner1=2;
       if(corner2==1) corner2=2;
       if(corner3==1) corner3=2;
     }


     if((corner1==0)&&(V.v[i-1].a[1]==-1)) {
      V.v[i]=get_next_vertex1(U,z,V.v[i-1],debug);
      corner1=1;
      V.v[i]=get_edge_leaders(U,z,V.v[i]);
     }


     if((corner2==0)&&(V.v[i-1].a[1]==-2)) {
      V.v[i]=get_next_vertex2(U,z,V.v[i-1],debug);
      corner2=1;
      V.v[i]=get_edge_leaders(U,z,V.v[i]);
     }

     if((corner3==0)&&(V.v[i-1].a[1]==-3)) {
      V.v[i]=get_next_vertex3(U,z,V.v[i-1],debug);
      corner3=1;
      V.v[i]=get_edge_leaders(U,z,V.v[i]);
     }


     printf("vertex %d: ",i);     cpt3(V.v[i]);

     /* ways to exit loop*/

     if((V.v[0].a[1]>0) && (corner1==1)) {
       if(dist(V.v[i],V.v[0])<.01) match=1;
     }

     if((V.v[0].a[1]<0) && (corner1==2)) {
       if(dist(V.v[i],V.v[0])<.01) match=1;
     }

     if((V.v[0].a[1]>0) && (corner2==1)) {
       if(dist(V.v[i],V.v[0])<.01) match=1;
     }

     if((V.v[0].a[1]<0) && (corner2==2)) {
       if(dist(V.v[i],V.v[0])<.01) match=1;
     }

     if((V.v[0].a[1]>0) && (corner3==1)) {
       if(dist(V.v[i],V.v[0])<.01) match=1;
     }

     if((V.v[0].a[1]<0) && (corner3==2)) {
       if(dist(V.v[i],V.v[0])<.01) match=1;
     }



     if(dist(V.v[i],V.v[0])<.000000001) match=1;

     ++i; 
     ++V.l;
     if(i>20) {
      printf("too many vertices\n");
      exit(1);
     }
   }

   for(i=1;i<=V.l;++i) {
     V.v[i].a[1]=V.v[i-1].a[3];
     V.v[i].a[2]=V.v[i-1].a[4];
   }
   V.v[V.l].a[3]=V.v[1].a[1];
   V.v[V.l].a[4]=V.v[1].a[2];


   /* special arrangement for vertices
      near the corners */

      for(i=1;i<=V.l;++i) {

	if((V.v[i].x>.95)&&(V.v[i].y<.05)) {
            V.v[i]=convert(1.0,0.0); V.v[i].a[1]=-99;
	}


	if((V.v[i].y>.95)&&(V.v[i].x<.05)) {
            V.v[i]=convert(0.0,1.0); V.v[i].a[1]=-99;
	}

	if((V.v[i].y>.95)&&(V.v[i].x>.95)) {
            V.v[i]=convert(1.0,1.0); V.v[i].a[1]=-99;
	}



      }
      if(V.l<3) exit(0);
   return(V);
}




/*this routine interpolates between two
  vertices along the edge which joins them.
  First the linear interpolation is made
  and then Newton's method is used to
  bring the point back to the edge.
  The method works well because the
  edges are pretty nearly straight.*/


complex interpolate_vertices(U,z1,z2,t)
     unfold U;
     complex z1,z2;
     double t;
{
  complex yy;
  complex y;
  int i;
  double test;
  complex grad;
  yy.x=(1.0-t)*z1.x+t*z2.x;
  yy.y=(1.0-t)*z1.y+t*z2.y;
  yy.a[1]=z2.a[1];
  yy.a[2]=z2.a[2];
  yy=newton_edge(U,yy,1);
  return(yy);
}






/*This routine finds a point, near the
  i_th edge, which should lie in the
  tile.  The routine computes
  -the interpolated center of the edge, X
  -the intersection of the left tangent
    line with the line joining CM and X.
  -the intersection of the right tangent
    line with the line joining CM and X.
   Then the routine chooses whichever
   of the three points is closest to CM.
   The success of the routine
   relies on the fact that the curvature
   of an edge of a tile is very nearly
   constant*/

complex inner_interpolate_vertices(U,V,i)
     unfold U;
   vertex_list V;
   int i;
{
  complex q[10];
  double d[5];
  q[1]=center_of_mass(V);
  q[2]=interpolate_vertices(U,V.v[i-1],V.v[i],0.5);
  q[3]=interpolate_vertices(U,V.v[i-1],V.v[i],0.00001);
  q[4]=interpolate_vertices(U,V.v[i-1],V.v[i],0.9999);
  q[5]=intersect(q[1],V.v[i-1],q[2],q[3]);
  q[6]=intersect(q[1],V.v[i],q[2],q[4]);

  d[1]=dist(q[1],q[2]);
  d[2]=dist(q[1],q[5]);
  d[3]=dist(q[1],q[6]);

  q[7]=q[2];
  if(d[2]<d[1]) q[7]=q[5];
  if((d[3]<d[2])&&(d[3]<d[1])) q[7]=q[6];
  return(q[7]);
}





/*Here is the newton plotting
  routine.  First the vertices are
  found and then the interpolation
  routine is used to draw the edges
  between the vertices.

  We make special arrangements for tiles
  which have a vertex near (1,0) or (0,1).
  For such tiles we just move the
  vertex to (0,1) or (1,0), as the
  case may be, and draw straight
  line segments to that vertex. Our
  experience is that all such
  tiles have this structure.
  The skeptical user can check
  the result against the basic plot.*/


void newton_draw(file,file2,ww,old_z,d)
     FILE *file,*file2;
     complex old_z;
     int d;
     word ww;
{
  vertex_list V;
  int d2;
  double t;
  int i,q,special;
  complex zz;
  complex z;
  double perimeter;
  word w;
  unfold U,U2;
  word w1;
  system("date");
  special=0;

  w=ww;
  U=edge_structure(w);
  z=new_center(w,old_z);
  V=get_vertex_list(U,w,z);

  fprintf(file2,"%d\n",V.l);
  perimeter=0.0;
  for(i=0;i<V.l;++i) {
    perimeter=perimeter+dist(V.v[i],V.v[i+1]);

  
  }



  for(i=1;i<=V.l;++i) {
    special=1;
    if(V.v[i-1].a[1]==-99) special=0;
    if(V.v[i].a[1]==-99) special=0;

    if(special==0)  tcl_cpt2(file,V.v[i-1]);

    if(special==1) {
      d2=2+d*dist(V.v[i-1],V.v[i])/perimeter;
    for(q=0;q<=d2;++q) {
      t=q*1.0;
      t=t/d2;
      zz=interpolate_vertices(U,V.v[i-1],V.v[i],t);
      tcl_cpt2(file,zz);
    }}

     tcl_cpt2(file2,V.v[i]);
     fprintf(file2,"%d\n%d\n",V.v[i].a[1],V.v[i].a[2]);
  }
  system("date");
}










void memory_newton_draw(file,file2,ww,old_z,d,VV)
     FILE *file,*file2;
     complex old_z;
     int d;
     word ww;
     vertex_list VV;
{
  vertex_list V;
  int d2;
  double t;
  int i,q,special;
  complex zz;
  complex z;
  double perimeter;
  word w;
  unfold U,U2;
  word w1;
  special=0;
  w=ww;
  U=edge_structure(w);
  z=new_center(w,old_z);
  V=VV;

  fprintf(file2,"%d\n",V.l);
  perimeter=0.0;
  for(i=0;i<V.l;++i)     perimeter=perimeter+dist(V.v[i],V.v[i+1]);

  for(i=1;i<=V.l;++i) {
    special=1;
    if(V.v[i-1].a[1]==-99) special=0;
    if(V.v[i].a[1]==-99) special=0;

    if(special==0)  tcl_cpt2(file,V.v[i-1]);

    if(special==1) {

      /*first we recover the vertex leaders.
        This routine isn't perfect.  It just
        computes the leaders for the point
        which is halfway between two
        consecutive vertices.  Thus point
        unfortunately is not on the edge.
        However, it is pretty close. */

      zz=plus(V.v[i-1],V.v[i]);
      zz=times(zz,convert(0.5,0.0));
      U2=assign_geometry(U,zz);
      w1=get_all_leaders(U2,1);
      wpt(w1);
      V.v[i].a[1]=w1.n[1];
      V.v[i].a[3]=w1.n[1];
      w1=get_all_leaders(U2,2);
      wpt(w1);
      V.v[i].a[2]=w1.n[1];
      V.v[i].a[4]=w1.n[1];



    /*back to the main loop*/

      d2=2+d*dist(V.v[i-1],V.v[i])/perimeter;
    for(q=0;q<=d2;++q) {
      t=q*1.0;
      t=t/d2;
      zz=interpolate_vertices(U,V.v[i-1],V.v[i],t);
      tcl_cpt2(file,zz);
    }}

     tcl_cpt2(file2,V.v[i]);
     fprintf(file2,"%d\n%d\n",V.v[i].a[1],V.v[i].a[2]);
  }
  system("date");
}










/*this routine draws the convex hull of the vertices*/

void hull_draw(file,file2,ww,old_z)
     FILE *file,*file2;
     complex old_z;
     word ww;
{
  word w;
  vertex_list V;
  int i;
  complex z;  
  unfold U;

  w=ww;
  U=edge_structure(w);
  z=new_center(w,old_z);
  V=get_vertex_list(U,w,z);
  fprintf(file2,"%d\n",V.l);

  fprintf(file2,"%d\n",V.l);
  for(i=1;i<=V.l;++i) {
    tcl_cpt2(file,V.v[i]);  
    fprintf(file2,"%d\n%d\n",V.v[i].a[1],V.v[i].a[2]);
  }
}


void memory_hull_draw(file,file2,ww,old_z,VV)
     FILE *file,*file2;
     complex old_z;
     word ww;
     vertex_list VV;
{
  word w;
  vertex_list V;
  int i;
  complex z;  
  unfold U;

  w=ww;
  U=edge_structure(w);
  z=new_center(w,old_z);
  V=VV;
  fprintf(file2,"%d\n",V.l);

  fprintf(file2,"%d\n",V.l);
  for(i=1;i<=V.l;++i) {
    tcl_cpt2(file,V.v[i]);  
    fprintf(file2,"%d\n%d\n",V.v[i].a[1],V.v[i].a[2]);
  }
}







/*this routine draws a polygon, with twice
  the number of vertices of the convex hull
  of the vertices, which should lie inside
  the orbit tile.*/


void inner_hull_draw(file,file2,ww,old_z)
     FILE *file,*file2;
     complex old_z;
     word ww;
{
  word w;
  vertex_list V;
  int i;
  int special;
  complex zz;
  complex z;  
  unfold U;

  w=ww;
  U=edge_structure(w);
  z=new_center(w,old_z);
  V=get_vertex_list(U,w,z);
  fprintf(file2,"%d\n",V.l);

  for(i=1;i<=V.l;++i) {
    special=1;
    if(V.v[i-1].a[1]==-99) special=0;
    if(V.v[i].a[1]==-99) special=0;
    if(special==0)  tcl_cpt2(file,V.v[i-1]);
    if(special==1) {
      zz=inner_interpolate_vertices(U,V,i);
      tcl_cpt2(file,V.v[i-1]);
      tcl_cpt2(file,zz);  
      fprintf(file2,"%d\n%d\n",V.v[i].a[1],V.v[i].a[2]);
    }
  }
}


void memory_inner_hull_draw(file,file2,ww,old_z,VV)
     FILE *file,*file2;
     complex old_z;
     word ww;
     vertex_list VV;
{
  word w,w1;
  vertex_list V;
  int i;
  int special;
  complex zz;
  complex z;  
  unfold U,U2;

  w=ww;
  U=edge_structure(w);
  z=new_center(w,old_z);
  V=VV;
  fprintf(file2,"%d\n",V.l);

  for(i=1;i<=V.l;++i) {
    special=1;
    if(V.v[i-1].a[1]==-99) special=0;
    if(V.v[i].a[1]==-99) special=0;
    if(special==0)  tcl_cpt2(file,V.v[i-1]);
    if(special==1) {

  /*first we recover the vertex leaders.
        This routine isn't perfect.  It just
        computes the leaders for the point
        which is halfway between two
        consecutive vertices.  Thus point
        unfortunately is not on the edge.
        However, it is pretty close. */

      zz=plus(V.v[i-1],V.v[i]);
      zz=times(zz,convert(0.5,0.0));
      U2=assign_geometry(U,zz);
      w1=get_all_leaders(U2,1);
      wpt(w1);
      V.v[i].a[1]=w1.n[1];
      V.v[i].a[3]=w1.n[1];
      w1=get_all_leaders(U2,2);
      wpt(w1);
      V.v[i].a[2]=w1.n[1];
      V.v[i].a[4]=w1.n[1];
    /*back to the main loop*/
      zz=inner_interpolate_vertices(U,V,i);
      tcl_cpt2(file,V.v[i-1]);
      tcl_cpt2(file,zz);  
      fprintf(file2,"%d\n%d\n",V.v[i].a[1],V.v[i].a[2]);
    }
  }
}

