#include <float.h>
 
typedef struct
{
  double x,y;
}   
pcomplex;

typedef struct
{pcomplex z1,z2,z3;
}
ptriangle;

typedef struct
{
pcomplex z[3];
}
triangle;

//------------------------------------ ------------------
// Ppoint and phull structs

typedef struct ppoint
{ struct ppoint *next;
  struct ppoint *prev;
  pcomplex z;
  int side; // -1=left side, 1=right side 
  double length; // Stores the distance between this point and the next point
                 // Further this length is positive when the edge is an infinite edge  
                 // otherwise we store -length.
} *Ppoint;

typedef struct
{ Ppoint 
  last_left, 
    last_right, 
    current; 
  double area;
} phull;

//------------------------------------------------
// Matrix struct

typedef struct
{ 
  double m[2][3];
}
matrix;

typedef struct
{
  matrix m[3];
} 
reflection;

// -----------------------------------------------
// triangle functions

triangle triangle_from_ptriangle(P)
     ptriangle P;
{
  triangle T;
  T.z[0]=P.z1;
  T.z[1]=P.z2;
  T.z[2]=P.z3;
  return T;
}

pcomplex triangle_center(T)
     triangle *T;
{
  pcomplex z;
  z.x=(T->z[0].x+T->z[1].x+T->z[2].x)/3;
  z.y=(T->z[0].y+T->z[1].y+T->z[2].y)/3;
  return z;
}


// a should be a triangle[4]
void triangle_subdivide(T, a)
     triangle *T, a[];
{
  int i,j;
  for (i=0; i<3;i++){
    for (j=0;j<3;j++){
      a[i].z[j].x=(T->z[i].x+T->z[j].x)/2;
      a[i].z[j].y=(T->z[i].y+T->z[j].y)/2;
    }
  }
  for (i=0; i<3;i++){
    a[3].z[i].x=(T->z[(i+1)%3].x+T->z[(i+2)%3].x)/2;
    a[3].z[i].y=(T->z[(i+1)%3].y+T->z[(i+2)%3].y)/2;
  }
}

double triangle_area(T)
     triangle *T;
{
  return 
    T->z[0].x*T->z[1].y-T->z[0].y*T->z[1].x+
    T->z[1].x*T->z[2].y-T->z[1].y*T->z[2].x+
    T->z[2].x*T->z[0].y-T->z[2].y*T->z[0].x;    
}

// -----------------------------------------------
// SOME USEFUL FUNCTIONS:

//int phull_alloc=0, phull_free=0;

void setLength(p)
     Ppoint p;
{
  p->length=-p->side*p->next->side*
    sqrt((p->z.x-p->next->z.x)*(p->z.x-p->next->z.x)+
	 (p->z.y-p->next->z.y)*(p->z.y-p->next->z.y));
}


// Sets the lengths and computes the measure of the hull
void setArea(h)
     phull *h;
{
  Ppoint p;
  h->area=0;
  p=h->last_left->next;
  while (p!=h->last_left){
    setLength(p);
    h->area+=p->length;
    p=p->next;
  }
  setLength(p);
  h->area+=p->length;
  p=p->next;
}

// this initializes a phull
void phull_setup(h)
     phull *h;
{
  h->last_left=(Ppoint) malloc (sizeof(struct ppoint));
  h->last_right=(Ppoint) malloc (sizeof(struct ppoint));
  //phull_alloc+=2;
  h->last_left->side=-1;
  h->last_right->side=1;
  h->last_left->z.x=h->last_left->z.y=0;
  h->last_right->z.x=1; h->last_right->z.y=0;
  h->last_left->next=h->last_left->prev=h->last_right;
  h->last_right->next=h->last_right->prev=h->last_left;
}

Ppoint insert(p,z,side)
     Ppoint p; 
     pcomplex z;
     int side;
{
  Ppoint next;
  if (p==NULL){
    fprintf(stdout,"error in insert4\n");
    fflush(stdout);
  }    
  if (p->next==NULL){
    fprintf(stdout,"error in insert\n");
    fflush(stdout);
  }    
  if (p->next->prev==NULL){
    fprintf(stdout,"error in insert3\n");
    fflush(stdout);
  }
  next=p->next;
  p->next=(Ppoint) malloc (sizeof(struct ppoint));
  //phull_alloc++;
  if (p->next==NULL){
    fprintf(stdout,"error in insert2\n");
    fflush(stdout);
  }    
  p->next->z=z;
  p->next->side=side;
  p->next->prev=p;
  p->next->next=next;
  next->prev=p->next;
  setLength(p);          // to keep the area
  setLength(p->next);    // up to date
  return p->next;
}

Ppoint insert2(p,z,side,length)
     Ppoint p; 
     pcomplex z;
     int side;
     double length;
{
  Ppoint next;
  if (p==NULL){
    fprintf(stdout,"error in insert4\n");
    fflush(stdout);
  }    
  if (p->next==NULL){
    fprintf(stdout,"error in insert\n");
    fflush(stdout);
  }    
  if (p->next->prev==NULL){
    fprintf(stdout,"error in insert3\n");
    fflush(stdout);
  }
  next=p->next;
  p->next=(Ppoint) malloc (sizeof(struct ppoint));
  //phull_alloc++;
  if (p->next==NULL){
    fprintf(stdout,"error in insert2\n");
    fflush(stdout);
  }    
  p->next->z=z;
  p->next->side=side;
  p->next->prev=p;
  p->next->next=next;
  p->next->length=length;
  next->prev=p->next;
  return p->next;
}

void delete(p)
     Ppoint p;
{
  p->prev->next=p->next;
  p->next->prev=p->prev;
  free(p);
  //  phull_free++;
}

int delete_from_hull(h,p)
     phull *h; 
     Ppoint p;
{
  Ppoint prev=p->prev;
  h->area-=p->length+prev->length;
  if (p->next==p)
    return 1;
  if (h->last_left==p)
    h->last_left=h->last_left->prev;
  if (h->last_right==p)
    h->last_right=h->last_right->prev; 
  if (h->current==p)
    h->current=NULL;
  delete(p);
  //setLength(prev);
  //h->area+=prev->length;
  return 0;
}

//this copys a phull
void phull_copy(h,new)
     phull *h; 
     phull *new;
{
  Ppoint p;
  new->last_left=(Ppoint) malloc (sizeof(struct ppoint));
  //  phull_alloc++;
  new->last_left->side=h->last_left->side;
  new->last_left->z=h->last_left->z;
  new->last_left->prev=new->last_left->next=new->last_left;
  new->last_left->length=h->last_left->length;

  p=h->last_left->prev;
  new->last_right=NULL;
  new->current=NULL;
  new->area=0;
  while (p!=h->last_left){ // while left side
    insert2(new->last_left, p->z, p->side, p->length);
    if (p==h->last_right)
      new->last_right=new->last_left->next;
    if (p==h->current)
      new->current=new->last_left->next;
    //fprintf(stdout, "length old = %e...", p->length);
    //fprintf(stdout, "length new = %e\n", new->last_left->next->length);
    p=p->prev;
  }
  if (p==h->last_right)
    new->last_right=new->last_left->next;
  if (p==h->current)
    new->current=new->last_left->next;
  if (h->last_right==NULL){
    fprintf(stdout, "error finding last_right pointer for phull_copy\n");
    fflush(stdout);
  }
  new->area=h->area;
}

// this frees the memory used by a phull
void phull_destroy(h)
     phull *h;
{
  Ppoint p=h->last_left;
  Ppoint q=p->next;
  while (p!=q){
    q=q->next;
    free(q->prev);
    //    phull_free++;
  }
  free(p);
  //  phull_free++;
  h->last_left=h->last_right=h->current=NULL;
}


// left if side=-1, right if side=1
void phull_insert(h,z,side)
     phull *h;
     pcomplex z; 
     int side;
{
  if (side==1){ //right side
    h->area-=h->last_right->length;
    h->current=insert(h->last_right, z, side);
    h->last_right=h->last_right->next;
    h->area+=h->last_right->length+h->last_right->prev->length;
    //fprintf(stdout, "area new = %e...", h->area);
    //fprintf(stdout, "area should be = %e\n", h->area);
  }
  else {// (side==-1){ //left side
    h->area-=h->last_left->length;
    h->current=insert(h->last_left, z, side);
    h->last_left=h->last_left->next;
    h->area+=h->last_left->length+h->last_left->prev->length;
  }
}




//-------------------------------------------------------
// Matrix multiplication functions

// returns with m1 storing m1*m2 (euclidean transforms)
void timesEquals(m1,m2)
     matrix *m1, *m2;
{
  double row[3];
  row[0]=m1->m[0][0]*m2->m[0][0]+m1->m[0][1]*m2->m[1][0];
  row[1]=m1->m[0][0]*m2->m[0][1]+m1->m[0][1]*m2->m[1][1];
  row[2]=m1->m[0][0]*m2->m[0][2]+m1->m[0][1]*m2->m[1][2]+m1->m[0][2];
  m1->m[0][0]=row[0];
  m1->m[0][1]=row[1];
  m1->m[0][2]=row[2];
  row[0]=m1->m[1][0]*m2->m[0][0]+m1->m[1][1]*m2->m[1][0];
  row[1]=m1->m[1][0]*m2->m[0][1]+m1->m[1][1]*m2->m[1][1];
  row[2]=m1->m[1][0]*m2->m[0][2]+m1->m[1][1]*m2->m[1][2]+m1->m[1][2];
  m1->m[1][0]=row[0];
  m1->m[1][1]=row[1];
  m1->m[1][2]=row[2];
}

double pcomplex_dot(c1,c2)
     pcomplex c1,c2;
{
  return c1.x*c2.x+c1.y*c2.y;
}

pcomplex eucTimes(m1,z)
     matrix *m1; 
     pcomplex z;
{
  pcomplex ret;
  ret.x=m1->m[0][0]*z.x+m1->m[0][1]*z.y+m1->m[0][2];
  ret.y=m1->m[1][0]*z.x+m1->m[1][1]*z.y+m1->m[1][2];
  return ret;
}


// returns an array of reflections from a triangle
void reflect_from_ptriangle(T,r)
     ptriangle T; 
     reflection *r;
{
  int i,j;
  matrix temp1,temp2;
  double x,y,det;
  {
    temp1.m[0][0]=1; temp1.m[0][1]=0; temp1.m[0][2]=-T.z2.x;
    temp1.m[1][0]=0; temp1.m[1][1]=1; temp1.m[1][2]=-T.z2.y;
    x=T.z3.x-T.z2.x;
    y=T.z3.y-T.z2.y;
    temp2.m[0][0]=(x*x-y*y); temp2.m[0][1]=2*x*y; temp2.m[0][2]=0;
    temp2.m[1][0]=2*x*y; temp2.m[1][1]=-(x*x-y*y); temp2.m[1][2]=0;
    det=sqrt(-(temp2.m[0][0]*temp2.m[1][1]-
	       temp2.m[0][1]*temp2.m[1][0]));
    for (i=0; i<2; i++)
      for (j=0; j<3; j++)
	temp2.m[i][j]*=1/det;
    r->m[0].m[0][0]=1; r->m[0].m[0][1]=0; r->m[0].m[0][2]=T.z2.x; 
    r->m[0].m[1][0]=0; r->m[0].m[1][1]=1; r->m[0].m[1][2]=T.z2.y; 
    timesEquals(&(r->m[0]),&temp2);
    timesEquals(&(r->m[0]),&temp1);
  }    
  {
    temp1.m[0][0]=1; temp1.m[0][1]=0; temp1.m[0][2]=-T.z3.x;
    temp1.m[1][0]=0; temp1.m[1][1]=1; temp1.m[1][2]=-T.z3.y;
    x=T.z1.x-T.z3.x;
    y=T.z1.y-T.z3.y;
    temp2.m[0][0]=(x*x-y*y); temp2.m[0][1]=2*x*y; temp2.m[0][2]=0;
    temp2.m[1][0]=2*x*y; temp2.m[1][1]=-(x*x-y*y); temp2.m[1][2]=0;
    det=sqrt(-(temp2.m[0][0]*temp2.m[1][1]-
	       temp2.m[0][1]*temp2.m[1][0]));
    for (i=0; i<2; i++)
      for (j=0; j<3; j++)
	temp2.m[i][j]*=1/det;
    r->m[1].m[0][0]=1; r->m[1].m[0][1]=0; r->m[1].m[0][2]=T.z3.x; 
    r->m[1].m[1][0]=0; r->m[1].m[1][1]=1; r->m[1].m[1][2]=T.z3.y; 
    timesEquals(&(r->m[1]),&temp2);
    timesEquals(&(r->m[1]),&temp1);
  }    
  {
    temp1.m[0][0]=1; temp1.m[0][1]=0; temp1.m[0][2]=-T.z1.x;
    temp1.m[1][0]=0; temp1.m[1][1]=1; temp1.m[1][2]=-T.z1.y;
    x=T.z2.x-T.z1.x;
    y=T.z2.y-T.z1.y;
    temp2.m[0][0]=(x*x-y*y); temp2.m[0][1]=2*x*y; temp2.m[0][2]=0;
    temp2.m[1][0]=2*x*y; temp2.m[1][1]=-(x*x-y*y); temp2.m[1][2]=0;
    det=sqrt(-(temp2.m[0][0]*temp2.m[1][1]-
	      temp2.m[0][1]*temp2.m[1][0]));
    for (i=0; i<2; i++)
      for (j=0; j<3; j++)
	temp2.m[i][j]*=1/det;
    r->m[2].m[0][0]=1; r->m[2].m[0][1]=0; r->m[2].m[0][2]=T.z1.x; 
    r->m[2].m[1][0]=0; r->m[2].m[1][1]=1; r->m[2].m[1][2]=T.z1.y; 
    timesEquals(&(r->m[2]),&temp2);
    timesEquals(&(r->m[2]),&temp1);
  }    
}

// returns an array of reflections from a triangle
void reflect_from_triangle(T,r)
     triangle T; 
     reflection *r;
{
  int i,j,k;
  matrix temp1,temp2;
  double x,y,det;
  for (k=0;k<3;k++){
    temp1.m[0][0]=1; temp1.m[0][1]=0; temp1.m[0][2]=-T.z[(k+1)%3].x;
    temp1.m[1][0]=0; temp1.m[1][1]=1; temp1.m[1][2]=-T.z[(k+1)%3].y;
    x=T.z[(k+2)%3].x-T.z[(k+1)%3].x;
    y=T.z[(k+2)%3].y-T.z[(k+1)%3].y;
    temp2.m[0][0]=(x*x-y*y); temp2.m[0][1]=2*x*y; temp2.m[0][2]=0;
    temp2.m[1][0]=2*x*y; temp2.m[1][1]=-(x*x-y*y); temp2.m[1][2]=0;
    det=sqrt(-(temp2.m[0][0]*temp2.m[1][1]-
	       temp2.m[0][1]*temp2.m[1][0]));
    for (i=0; i<2; i++)
      for (j=0; j<3; j++)
	temp2.m[i][j]*=1/det;
    r->m[k].m[0][0]=1; r->m[k].m[0][1]=0; r->m[k].m[0][2]=T.z[(k+1)%3].x; 
    r->m[k].m[1][0]=0; r->m[k].m[1][1]=1; r->m[k].m[1][2]=T.z[(k+1)%3].y; 
    timesEquals(&(r->m[k]),&temp2);
    timesEquals(&(r->m[k]),&temp1);
  }    
}

void matrix_identity(m)
  matrix *m;
{
  m->m[0][0]=m->m[1][1]=1;
  m->m[0][1]=m->m[0][2]=m->m[1][0]=m->m[1][2]=0;
}

int det5(a,b,c)
     pcomplex *a, *b, *c;
{
  doubleList p[6];
  doubleList dl;
  int ret;

  p[0]=storeProduct(a->x, b->y);
  p[1]=storeProduct(b->x, c->y);
  p[2]=storeProduct(c->x, a->y);
  p[3]=storeProduct(-a->x, c->y);
  p[4]=storeProduct(-b->x, a->y);
  p[5]=storeProduct(-c->x, b->y);
  dl=DLsumDestroy(p,6);
  DLmag(&dl);
  DLsort(&dl);
  ret=DLsign(&dl);
  DLdestroy(&dl);
  return ret;
}    

// returns the sign of the determinant, that is the orientation of the triangle
int det(x,y,z)
     pcomplex *x, *y, *z;
{
  double det=0,
    error=0,
    sum;
  int exp;

  sum=x->x*y->y;
  if (sum!=0){
    frexp(sum,&exp);
    error+=ldexp(1.,exp-DIGITS);
  }
  det+=sum;
  sum=x->y*z->x;
  if (sum!=0){
    frexp(sum,&exp);
    error+=ldexp(1.,exp-DIGITS);
  }
  det+=sum;
  if (det!=0){
    frexp(det,&exp);
    error+=ldexp(1.,exp-DIGITS);
  }
  sum=y->x*z->y;
  if (sum!=0){
    frexp(sum,&exp);
    error+=ldexp(1.,exp-DIGITS);
  }
  det+=sum;
  if (det!=0){
    frexp(det,&exp);
    error+=ldexp(1.,exp-DIGITS);
  }
  sum=-x->x*z->y;
  if (sum!=0){
    frexp(sum,&exp);
    error+=ldexp(1.,exp-DIGITS);
  }
  det+=sum;
  if (det!=0){
    frexp(det,&exp);
    error+=ldexp(1.,exp-DIGITS);
  }
  sum=-x->y*y->x;
  if (sum!=0){
    frexp(sum,&exp);
    error+=ldexp(1.,exp-DIGITS);
  }
  det+=sum;
  if (det!=0){
    frexp(det,&exp);
    error+=ldexp(1.,exp-DIGITS);
  }
  sum=-y->y*z->x;
  if (sum!=0){
    frexp(sum,&exp);
    error+=ldexp(1.,exp-DIGITS);
  }
  det+=sum;
  if (det!=0){
    frexp(det,&exp);
    error+=ldexp(1.,exp-DIGITS);
  }
  if (det>0){
    if (det<error)
      return det5(x,y,z);
    else 
      return 1;
  }  
  if (det<0){
    if (det>-error)
      return det5(x,y,z);
    else 
      return -1;
  }
  return det5(x,y,z);
}

// -----------------------------------------------
// SOME USEFUL FUNCTIONS:


//--------------------------------------------
// INPUT AND OUTPUT MEATHODS

void Ppoint_print_file(file,p)
     FILE *file;
     Ppoint p;
{
  fprintf(file,"(%8.16lf, %8.16lf) %d\n", p->z.x, p->z.y, p->side);
}

void phull_print_file(file,h)
     FILE *file;
     phull *h;
{
  Ppoint p=h->last_left,q=p->next;
  fprintf(file,"BEGIN phull\n");
  while (q!=p){
    Ppoint_print_file(file,q);
    q=q->next;
  }
  Ppoint_print_file(file,q);
  q=q->next;

  fprintf(file,"END phull\n");

}

// print out the pword
void matrix_print_file(file,m)
     FILE *file;
     matrix *m;
{
  fprintf(file,"[%8.16lf %8.16lf %8.16lf]\n",m->m[0][0],m->m[0][1],m->m[0][2]);
  fprintf(file,"[%8.16lf %8.16lf %8.16lf]\n",m->m[1][0],m->m[1][1],m->m[1][2]);
}

void pcomplex_tri_print_file(file,z)
     FILE *file;
     pcomplex z;
{
  fprintf(file,"(%8.8lf Pi, %8.8lf Pi, %8.8lf Pi)\n",z.x,z.y,1-(z.x+z.y));
}

