/*
 * RationalTriangle.java
 *
 * Created on December 9, 2005, 6:22 PM
 */

import java.awt.geom.*;

/**
 * The integers (x,y,z) represent the triangle with angles
 * (x/(x+y+z)*pi, y/(x+y+z)*pi, z/(x+y+z)*pi).
 * @author  pat
 */
public class RationalTriangle {
    // array of coefficients of lines as described above.
    int[] x=new int[3];
    
    /** Creates a new instance of RationalTriangle */
    public RationalTriangle(int a, int b, int c) {
        x[0]=a;
        x[1]=b;
        x[2]=c;
        reduce();
    }
    
    /** Construct the intersection of two rational lines */
    public RationalTriangle(RationalLine a, RationalLine b) {
        x=NumberTheory.cross(a.x,b.x);
        reduce();
    }
    
    /** returns the rational triangle with angles (p1/q1*(pi/2), p2/q2*(pi/2)) */
    public static RationalTriangle fromMcbRational(int p1, int q1,int p2, int q2){
        return new RationalTriangle(p1*q2,p2*q1, 2*q1*q2-p1*q2-p2*q1);
    }

    
    /** returns an array {p1,q1,p2,q2} */
    public int[] toMcbRational(){
        int[] ret=new int[4];
        ret[0]=2*x[0];
        ret[1]=x[0]+x[1]+x[2];
        ret[2]=2*x[1];
        ret[3]=x[0]+x[1]+x[2];
        int g=NumberTheory.gcd(ret[0],ret[1]);
        if (g!=0) {
            ret[0]/=g;
            ret[1]/=g;
        }
        g=NumberTheory.gcd(ret[2],ret[3]);
        if (g!=0) {
            ret[2]/=g;
            ret[3]/=g;
        }
        return ret;
    }
    
    /** returns "(p1/q1, p2/q2)" */
    public String toMcbString(){
        int[] s=toMcbRational();
        return "("+s[0]+"/"+s[1]+", "+s[2]+"/"+s[3]+")";
    }
    
    public void reduce() {
        int gcd=NumberTheory.gcd(x[0],NumberTheory.gcd(x[1],x[2]));
        if (gcd!=0) {
            x[0]/=gcd;
            x[1]/=gcd;
            x[2]/=gcd;
        }
        // normalize so that the last nonzero coordinate is positive.
        boolean change=false;
        for (int i=2; i>=0; i--) {
            if (x[i]<0) {
                change=true;
                break;
            }
            if (x[i]>0) {
                change=false;
                break;
            }
        }
        if (change)
            for (int i=2; i>=0; i--)
                x[i]*=-1;
    }
    
    
    /** Returns true if the triangle is a valid triangle.
     * The angles must all be positive, and the both the first and the second
     * angles are less than pi/2.
     */
    public boolean isValid() {
        if (x[2]<0) {
            for (int i=0; i<3; i++)
                x[i]*=-1;
        }
        return ((x[0]>0)&&(x[0]<x[2])&&(x[1]>0)&&(x[1]<x[2]));
    }
    
    /** Returns true if the triangle is a valid triangle or lies on the
     * boundary of the parameter space.
     */
    public boolean isBoundaryValid() {
        if (x[2]<0) {
            for (int i=0; i<3; i++)
                x[i]*=-1;
        }
        return ((x[0]>=0)&&(0<=-x[0]+x[1]+x[2])&&(x[1]>=0)&&(0<=x[0]-x[1]+x[2])&&
        ((x[0]!=0)||(x[1]!=0)||(x[2]!=0))); // and nonzero
    }
    
    public Complex toComplex() {
        return new Complex(2.0*x[0]/(x[0]+x[1]+x[2]), 2.0*x[1]/(x[0]+x[1]+x[2]));
    }
    
    public Point2D.Double toPoint() {
        return new Point2D.Double(2.0*x[0]/(x[0]+x[1]+x[2]), 2.0*x[1]/(x[0]+x[1]+x[2]));
    }

    public String toString() {
        return "("+x[0]+", "+x[1]+", "+x[2]+")";
    }

}
