/*
 * Tile.java
 *
 * Created on December 8, 2005, 7:26 AM
 */

import java.awt.*;
import java.awt.geom.*;

/**
 * This interface represents a generic tile. 
 * They may be drawn however you like. Additionally, they can store any data 
 * you might want them to.
 *
 * Since there are many different types of tiles (a tile is anything that 
 * implements this Tile interface), we need a way to distinguish them. Java
 * has a way to tell what class your tile really is.
 *
 * Suppose you want to check to see if the tile is a DyadicTile and do something
 * with the Dyadic tile if it is. You can do this with code like this:
 *
 * //(suppose T is a Tile)
 * if (T.getClass().getName()=="DyadicTile") { // check to see if the tile is a DyadicTile
 *     DyadicTile DT=(DyadicTile)(T); // convert T to a DyadicTile we call DT
 *     DT.doSomething();
 * }
 *
 * @author  pat
 */
public interface Tile {
    
    /** return the word for this tile as a string */
    public String getStringWord();
    
    /** return true if the tile can be drawn */
    public boolean isPaintable();
    
    /** set the color of this tile */
    public void setColor(Color C);
    
    /** Return a rectangle containing the tile*/
    public Rectangle2D getBounds();
    
    /** Return true if the tile contains the point (x,y) */
    public boolean contains(double x, double y);
    
    /** Paint the tile on to the graphics object. Coordinates used are
     * McBilliards Coordinates. The output must be modified by T.
     *
     * For example to draw the tile for the Fagnano Curve, you might use:
     * GeneralPath gp=new GeneralPath();
     * gp.moveTo((float)1,(float)0);
     * gp.lineTo((float)1,(float)1);
     * gp.lineTo((float)0,(float)1);
     * gp.closePath();
     * g.fill(T.createTransformedShape(GP));
     */
    public void paint(Graphics2D g, AffineTransform T);
    
    /** Paint the tile to indicate that the tile is selected */
    public void paintSelected(Graphics2D g, AffineTransform T);
    
    /** Return true if the tile can be saved to a file */
    public boolean isSaveable();
    
    /** Return a string that can be used to save this tile to a file */
    public String saveString();
}
