import java.applet.Applet;
import java.awt.event.*;
import java.awt.*;
import java.awt.geom.*;

public class TetraCover {

    /**The purpose of this class is to furnish the routines that go into
the proof that our partition of the unit torus by tetrahedra really is
a partition.  The proof works like this.  There are 11 heights where
vertices occur.  We choose 10 values interlacing these heights.  For
each such height, we slice the torus with a horizontal plane of that
height.  We then show that the resulting polygons partition the 2-torus
slice.  The method is as follows:

1. We find all edges that cross the slice.

2. For each edge, we compute the point where the edge crosses.

3. We find all tetrahedra which contain this point on their boundary. 
   (Since the tetrahedra have disjoint interiors, the point is not an
   interior point of any tetrahedron.  Call these tetrahedra the
   involved tetrahedra.

4. Once we have the list of tetrahedra involved with an edge, we
   compute the link of the edge.  We check that the link is a
   circle for all edges.

    **/



    /**FINDING THE CROSSING EDGES.  Here t is the height.**/

    public static GoldenPolyhedron[] hitsLevel(GoldenReal t,GoldenPolyhedron[] P) {
	GoldenPolyhedron[] LIST=new GoldenPolyhedron[0];
	for(int i=0;i<P.length;++i) {
	    LIST=concatenate(LIST,hitsLevel(t,P[i]));
	}
	return(LIST);
    }





    public static GoldenPolyhedron[] hitsLevel(GoldenReal t,GoldenPolyhedron P) {
	GoldenPolyhedron[] LIST=new GoldenPolyhedron[6];
	int total=0;
	GoldenVector[] X=new GoldenVector[2];
	for(int i=0;i<4;++i) {
	    for(int j=0;j<4;++j) {
		X[0]=new GoldenVector(P.V[i]);
		X[1]=new GoldenVector(P.V[j]);
		boolean test1=GoldenReal.isLess(X[0].x[2],t);
		boolean test2=GoldenReal.isLess(t,X[1].x[2]);
		if((test1==true)&&(test2==true)) {
		    LIST[total]=new GoldenPolyhedron();
		    LIST[total].count=2;
		    LIST[total].V[0]=new GoldenVector(X[0]);
		    LIST[total].V[1]=new GoldenVector(X[1]);
		    ++total;
		}
	    }
	}
	GoldenPolyhedron[] LIST2=new GoldenPolyhedron[total];
	for(int i=0;i<total;++i) LIST2[i]=new GoldenPolyhedron(LIST[i]);
	return(LIST2);
    }


    /**concatenates two lists of polyhedra, ignoring redundancies**/

    public static GoldenPolyhedron[] concatenate(GoldenPolyhedron[] LIST1,GoldenPolyhedron[] LIST2) {
	if(LIST2==null) return(LIST1);
	if(LIST1==null) return(LIST2);
	GoldenPolyhedron[] LIST3=new GoldenPolyhedron[LIST1.length+LIST2.length];
	for(int i=0;i<LIST1.length;++i) LIST3[i]=new GoldenPolyhedron(LIST1[i]);
	int n=LIST1.length;
	for(int i=0;i<LIST2.length;++i) LIST3[n+i]=new GoldenPolyhedron(LIST2[i]);
	return(LIST3);
    }

    /**concatenates two lists of vectors, ignoring redundancies**/

    public static GoldenVector[] concatenate(GoldenVector[] LIST1,GoldenVector[] LIST2) {
	if(LIST2==null) return(LIST1);
	if(LIST1==null) return(LIST2);
	GoldenVector[] LIST3=new GoldenVector[LIST1.length+LIST2.length];
	for(int i=0;i<LIST1.length;++i) LIST3[i]=new GoldenVector(LIST1[i]);
	int n=LIST1.length;
	for(int i=0;i<LIST2.length;++i) LIST3[n+i]=new GoldenVector(LIST2[i]);
	return(LIST3);
    }


    /**COMPUTING THE INTERSECTION POINT**/

    /**finds the intersection point of e with the slice of height t.
       Actually, since we want to work over Z[phi], we clear denominators,
       and actually compute a scaled-up version of this point.  When we
       test this point against our polyhedron P, we instead use a similarly
       scaled version of P -- namely Q.**/

    public static GoldenVector interpolate(GoldenReal t,GoldenPolyhedron e) {
	GoldenVector V=new GoldenVector();
	GoldenReal x0=new GoldenReal(e.V[0].x[0]);
	GoldenReal x1=new GoldenReal(e.V[1].x[0]);
	GoldenReal y0=new GoldenReal(e.V[0].x[1]);
	GoldenReal y1=new GoldenReal(e.V[1].x[1]);
	GoldenReal z0=new GoldenReal(e.V[0].x[2]);
	GoldenReal z1=new GoldenReal(e.V[1].x[2]);
	GoldenReal s0=GoldenReal.minus(t,z0);
	GoldenReal s1=GoldenReal.minus(z1,z0);
	V.x[0]=GoldenReal.plus(GoldenReal.times(s1,x0),GoldenReal.times(s0,GoldenReal.minus(x1,x0)));
	V.x[1]=GoldenReal.plus(GoldenReal.times(s1,y0),GoldenReal.times(s0,GoldenReal.minus(y1,y0)));
	V.x[2]=GoldenReal.times(s1,t);
	return(V);
    }


    /**FINDING THE INVOLVED TETRAHEDRA.**/

    public static GoldenVector[] orthoMatch(GoldenReal t,GoldenPolyhedron e,GoldenPolyhedron[] P) {
	GoldenVector[] LIST=new GoldenVector[0];
	for(int i=0;i<P.length;++i) {
	    for(int j=-1;j<=1;++j) {
		for(int k=-1;k<=1;++k) {
		  int[] A={2*j,0,2*k,0,0,0};
	          GoldenVector W=new GoldenVector(A);
	          GoldenPolyhedron X=P[i].translate(W);
                  LIST=concatenate(LIST,orthoMatch(t,e,X));
		}
	    }
	}
	return(LIST);
    }

    /**This routine checks if the tetrahedron P is involved with the edge e. If so, 
       there are two possibilities:

       1. e is collinear with an edge of P.  In this case, e is incident to 2
          faces of P.  We choose 2 vectors that lie in these incidentg faces and
          are perpendicular to e.

       2. the point of e which intersects our slice lies in the interior of a face
          of P.  in this case, we return two antipodal vectors that lie in
          this face and are perpendicular to e.
 
    **/


    public static GoldenVector[] orthoMatch(GoldenReal t,GoldenPolyhedron e,GoldenPolyhedron P) {
	int[] u=match(t,e,P);
	if(u==null) return(null);
	if(u.length==1) return(null);
	if(u.length==4) return(null);
        GoldenVector V=GoldenVector.minus(e.V[1],e.V[0]);
        GoldenVector VV=GoldenVector.minus(P.V[u[0]],P.V[u[1]]);
	/**point lies on an edge**/
	if(u.length==2) {
	    GoldenVector bad=GoldenVector.cross(V,VV);
	    if(bad.isZero()==false) return(null);
	    int[] v=complement(u);
	    GoldenVector V1=GoldenVector.ortho(V,GoldenVector.minus(P.V[v[0]],P.V[u[0]]));
	    GoldenVector V2=GoldenVector.ortho(V,GoldenVector.minus(P.V[v[1]],P.V[u[0]]));
	    GoldenVector[] LIST={V1,V2};
	    return(LIST);
	}

	/**point lies on a face**/
	GoldenVector W=GoldenVector.minus(P.V[u[0]],P.V[u[1]]);
	GoldenVector V1=GoldenVector.ortho(V,W);
	if(V1.isZero()==true) {
             W=GoldenVector.minus(P.V[u[0]],P.V[u[2]]);
             V1=GoldenVector.ortho(V,W);
	}
	GoldenVector V2=V1.negate();
        GoldenVector[] LIST={V1,V2};
	return(LIST);
    }


    /**This determines the indices of P corresponding to the edge or face
       that contains (e intersect slice).  If there are no such indices,
       then null is returned.**/

    public static int[] match(GoldenReal t,GoldenPolyhedron e,GoldenPolyhedron P) {
	GoldenReal s=GoldenReal.minus(e.V[1].x[2],e.V[0].x[2]);
	GoldenPolyhedron Q=P.scale(s);
	GoldenVector W=interpolate(t,e);
	int[] u=locate(W,Q);
	return(u);
    }



    /**if the indices of u are 1,2, the routine returns 3,4
       if the indices of u are 2,3, the routine returns 1,4, etc. **/

    public static int[] complement(int[] u) {
	int total=0;
	int[] v=new int[2];
	for(int i=0;i<4;++i) {
	    if((i!=u[0])&&(i!=u[1])) {
		v[total]=i;
		++total;
	    }
	}
	return(v);
    }




    /**This routine locates a point relative to a polyhedron.
       We only use the routine when the polyhedron is a tetrahedron.
 
       a return of null means that the point lies outside the tetrahedron
       a return of {i} means that the point is the ith vertex
       a return of {i,j} means that the point is in the interior of the (i,j)th edge.
       a return of {i,j,k} means that the point is in the interior of the (i,j,k)th face.
       a return of {0,1,2,3} means that the point is in the interior.
    **/

    public static int[] locate(GoldenVector P,GoldenPolyhedron G) {
	GoldenVector[] T=new GoldenVector[4];
	for(int i=0;i<4;++i) T[i]=new GoldenVector(G.V[i]);

	/*make the 4 inside tetrahedra*/
	GoldenVector PPPP=new GoldenVector(P);
	GoldenVector[] T0={PPPP,T[1],T[2],T[3]};
	GoldenVector[] T1={T[0],PPPP,T[2],T[3]};
	GoldenVector[] T2={T[0],T[1],PPPP,T[3]};
	GoldenVector[] T3={T[0],T[1],T[2],PPPP};

	/*compute the GoldenVector.volumes*/
	GoldenReal r=GoldenVector.volume(T);
	GoldenReal r0=GoldenVector.volume(T0);
	GoldenReal r1=GoldenVector.volume(T1);
	GoldenReal r2=GoldenVector.volume(T2);
	GoldenReal r3=GoldenVector.volume(T3);

	/**check outside condition*/
	GoldenReal s=new GoldenReal(0,0);
	s=GoldenReal.plus(s,r0);
	s=GoldenReal.plus(s,r1);
	s=GoldenReal.plus(s,r2);
	s=GoldenReal.plus(s,r3);
	if(GoldenReal.equals(r,s)==false) return(null);

	/**find location*/
	boolean[] z={r0.isZero(),r1.isZero(),r2.isZero(),r3.isZero()};

	int[] u=new int[4];
	int total=0;
	for(int i=0;i<4;++i) {
	    if(z[i]==false) {
		u[total]=i;
		++total;
	    }
	}
	int[] v=new int[total];
	for(int i=0;i<total;++i) v[i]=u[i];
	return(v);
    }


    /**CHECKING THE LINK**/

    public static boolean link(GoldenVector[] LIST) {
	if(circleLink(LIST)==true) return(true);
	if(halfCircleLink(LIST)==true) return(true);
	return(false);
    }

    public static boolean circleLink(GoldenVector[] LIST) {
	for(int i=0;i<LIST.length;++i) {
	    boolean check=false;
	    for(int j=0;j<LIST.length;++j) {
		GoldenVector V1=GoldenVector.cross(LIST[i],LIST[j]);
		GoldenReal d=GoldenVector.dot(LIST[i],LIST[j]);
		if((i!=j)&&(d.isPositive()==true)&&(V1.isZero()==true)) check=true;
	    }
	    if(check==false) return(false);
	}
	return(true);
    }


    public static boolean halfCircleLink(GoldenVector[] LIST) {
	int total=0;
	for(int i=0;i<LIST.length;++i) {
	    boolean check=false;
	    for(int j=0;j<LIST.length;++j) {
		GoldenVector V1=GoldenVector.cross(LIST[i],LIST[j]);
		GoldenReal d=GoldenVector.dot(LIST[i],LIST[j]);
		if((i!=j)&&(d.isPositive()==true)&&(V1.isZero()==true)) check=true;
	    }
	    if(check==false) ++total;
	}

	/**should be 2 mismatches, coming from antipodal vectors*/
	if(total!=2) return(false);

	for(int i=0;i<LIST.length;++i) {
	    boolean check=false;
	    for(int j=0;j<LIST.length;++j) {
		GoldenVector V1=GoldenVector.cross(LIST[i],LIST[j]);
		if((i!=j)&&(V1.isZero()==true)) check=true;
	    }
        if(check==false) return(false);
	}
	return(true);
    }

}


