import java.applet.Applet;
import java.awt.event.*;
import java.awt.*;
import java.awt.geom.*;
import java.math.*;


public class StripRenorm {

    /**This class contains routines that deal with the
       calculations that go into the proof of the
       Fundamental Orbit Theorem and the 
       Near Reduction Theorem.  That is, they deal with
       the A-cores and B-cores that lie in the strip Sigma.*/

    /**In these routines the point u is assumed to be an A-marker.
       That is, theta(u) lies in the A-renorm set.  This routine
       renormalizes u and then takes the point in the band that
       minimizes the horizontal distance to the origin.**/

    public static Complex renorm(Complex u,boolean reverse) {
	Vector VA=TorusMap.theta(u);
        Vector VB=mapAtoB(VA);
	Complex z=matchB(VB);
	z=minimizeB(z,reverse);
	return(z);
    }


    /**This renormalizes u and returns all the values in the
       B-band that is the renormalization of the A-band containing u.*/

    public static Complex[] renormBand(Complex u,boolean reverse) {
	Vector VA=TorusMap.theta(u);
        Vector VB=mapAtoB(VA);
	Complex z=matchB(VB);
	return(bandListB(z,reverse));
    }

    /**This routine searches through the A-core of u, and finds the
       point that minimizes the horizontal distance to the origin.*/

    public static Complex minimizeA(Complex u) {
	Complex[] LIST=bandListA(u);
	double min=1000;
	Complex z2=new Complex(u);
	for(int i=0;i<LIST.length;++i) {
            double test=Math.abs(LIST[i].x);
	    if(min>test) {
		min=test;
		z2=new Complex(LIST[i]);
	    }
	}
	return(z2);
    }

    /**This gets the full list of all the points in the A-core of u.**/

    public static Complex[] bandListA(Complex u) {
	Complex[] LIST=new Complex[1000];
	PinwheelMap PIN=new PinwheelMap();
	Complex z1=new Complex(u);
	boolean in=false;
	int count=0;
	while((count<1000)&&(in==false)) {
	    LIST[count]=new Complex(z1);
	    ++count;
     	    z1=PIN.pi(z1);
	    in=testInside(z1,true);
	}
	Complex[] LIST2=new Complex[count];
	for(int i=0;i<count;++i) LIST2[i]=new Complex(LIST[i]);
	return(LIST2);
    }


    /**In these routines v is such that Theta(v) lies in the
       B renorm set.**/

    /**This routine searches through the points in the B-core
       of v and finds the one that minimizes the horizontal
       distance to the origin.*/

    public static Complex minimizeB(Complex v,boolean reverse) {
	Complex[] LIST=bandListB(v,reverse);
	double min=1000;
	Complex z2=new Complex(v);
	for(int i=0;i<LIST.length;++i) {
            double test=Math.abs(LIST[i].x);
	    if(min>test) {
		min=test;
		z2=new Complex(LIST[i]);
	    }
	}
	return(z2);
    }


    /**This routine returns all the points in the B-core
       of v.**/

    public static Complex[] bandListB(Complex v,boolean reverse) {
	Complex[] LIST=new Complex[1000];
	PinwheelMap PIN=new PinwheelMap();
	Complex z1=new Complex(v);
	boolean in=false;
	int count=0;
	while((count<1000)&&(in==false)) {
	    LIST[count]=new Complex(z1);
	    ++count;
     	    if(reverse==false) z1=PIN.pi(z1);
	    if(reverse==true)  z1=PIN.piInverse(z1);
	    in=testInside(z1,false);
	}
	Complex[] LIST2=new Complex[count];
	for(int i=0;i<count;++i) LIST2[i]=new Complex(LIST[i]);
	return(LIST2);
    }


    /**These routines map the A-renorn set to the B-renorm set.**/

    public static Vector mapAtoB(Vector V) {
        int[] test=DataRenorm.indexA(V);
	if(test==null) return(null);
	Vector W=DataRenorm.AtoB(test[0],test[1],test[2],V);
	return(W);
    }



    public static Polyhedron mapAtoB(Polyhedron P) {
	Vector V=P.getCenter();
        int[] test=DataRenorm.indexA(V);
	Polyhedron Q=new Polyhedron();
	Q.count=P.count;
	for(int i=0;i<P.count;++i) {
	    Q.V[i]=DataRenorm.AtoB(test[0],test[1],test[2],P.V[i]);
	}
	return(Q);
    }




    /**This routine attempts to find a point z such that
       theta(z)=V.  This routine is used in times when
       it is known that such a z exists.*/

    public static Complex matchB(Vector V) {
	for(int k=0;k<1000;++k) {
	    for(int q=0;q<=1;++q) {
	      Complex z=matchB(V,2*k,q);
	      if(z!=null) return(z);
	      z=matchB(V,-2*k,q);
	      if(z!=null) return(z);
	    }
	}
	return(null);
    }

    /**This routine is the same as the previous one,
       except that it only searches over a certain 
       segment in order to find z.  So, the previous
       routine invokes this one for a succession of
       segments.**/

    public static Complex matchB(Vector V,int k,int shift) {
	double p=GoldenRatio.phi(1);
	Complex z1=new Complex((k-1)*p,V.x[2]-2*shift);
	Complex z2=new Complex((k+1)*p,V.x[2]-2*shift);
	Vector W1=TorusMap.theta(z1);
	Vector W2=TorusMap.theta(z2);
	if(W1.x[0]>1) W1.x[0]=W1.x[0]-2;
	if(W2.x[0]<1) W2.x[0]=W2.x[0]+2;
 	if(W2.x[1]<W1.x[1]) W2.x[1]=W2.x[1]+2;
	for(int s=-2;s<=2;s=s+2) {
	    Vector V1=Vector.plus(W1,new Vector(0,s,0));
	    Vector V2=Vector.plus(W2,new Vector(0,s,0));
	    int test=Vector.between(V1,V,V2);
	    if(test==1) {
		double h=V.x[0]*p;
		z1=Complex.plus(z1,new Complex(h,0));
		return(z1);
	    }
	}
	return(null);
    }


    /**This routine tests whether a given complex number
       in the strip maps into the A-set or the B-set.
       When A=true, we use the A set. When A=false,
       we use the B set.**/

    public static boolean testInside(Complex z,boolean A) {
        Vector V=TorusMap.theta(z);
	boolean test=false;
	if(A==true) test=DataRenorm.insideA(V);
	if(A==false) test=DataRenorm.insideB(V);
	return(test);
    }

    /**This map applies either the forward or backward pinwheel map until
       the given point lies in A or B.  The variable (force) determines
       whether we apply the map before checking for containment.  It suffices
       to check the first 800 iterates, by the renormalization theorem.**/

    public static Complex inside(Complex z,boolean force,boolean forward,boolean A) {
	PinwheelMap PIN=new PinwheelMap();
	if(force==false) {
	    boolean test=testInside(z,A);
	    if(test==true) return(z);
	}
	for(int i=0;i<800;++i) {
             if(forward==false) z=PIN.piInverse(z);
	     if(forward==true)  z=PIN.pi(z);
	     boolean test=testInside(z,A);
	     if(test==true) return(z);
	}
	return(null);
    }

}



