import java.applet.Applet;
import java.awt.*;
import java.awt.event.*;
import java.applet.*;
import java.awt.geom.*;
import java.math.*;
import java.io.*;


/**Say that an N-tile is a convex polygon P in the fundamental
   strip such that N iterates of the pinwheel map are entirely
   defined on the interior of P.  This routine takes a pair
   (P,N), where P is a golden polygon and N is an integer,
   and verifies rigorously that P is an N-tile.
   
   Since we have proved the compactification
   theorem, we use the 3D polyhedron exchange for the
   verification.  So, the verification works like this:

   1. we use floating point arithmetic to compute the
      itinerary of the center of mass of P.
      This gives us an integer  sequence I(N)
   2. We verify that all vertices have I(N)
      as a feasible itinerary.  This guarantees
      that all interior points have I(N) as the
      actual (and unique) itinerary.
**/



public class VerifyTile  {


    /**MAIN ROUTINES**/


    /**This routine verifies that P is a periodic orbit tile
       having period at most LIM.**/

    public static int verifyPeriodic(GoldenPolyWedge P,int LIM,boolean print) {
	int[] I=VerifySupport.getItinerary(P,LIM);
	boolean test=verify(P,I);
	if(test==false) return(-1);
	GoldenVector V=VerifySupport.getTotalMove(I);
	if(V.isZero()==false) return(-1);
   	if(print==true) {
	    System.out.println("periodic itinerary");
            Lists.printout(I);
	}
	return(I.length);
    }

    /**This routine verifies that a given GoldenComplex,
       when mapped into the torus, has the given itinerary
       as a feasible itinerary.*/

    public static boolean verify(GoldenPolyWedge P,int[] I) {
	for(int i=0;i<P.count;++i) {
	    GoldenVector V=VerifySupport.theta(P.z[i]);
	    boolean test=testFeasible(V,I);
	    if(test==false) return(false);
	}
	return(true);
    }

    /**This routine verifies that a given GoldenComplex,
       when mapped into the torus, has the given itinerary
       for the backwards dynamics.  The itinerary is given 
       in terms of the forward tiles.**/

    public static boolean verifyInverse(GoldenPolyWedge P,int[] I) {
	for(int i=0;i<P.count;++i) {
	    GoldenVector V=VerifySupport.theta(P.z[i]);
	    boolean test=testFeasibleInverse(V,I);
	    if(test==false) return(false);
	}
	return(true);
    }


     /**This routine is a bit tricky, because we have to handle
       the cases when W lies on the boundary of the fundamental
       domain.  So, we allow ourselves to move W0 by an element
       in (2Z)^3 in order to find the correct representative.
       The routine fails if we ever return a null vector.*/


    public static boolean testFeasible(GoldenVector W0,int[] I) {
	GoldenVector W=new GoldenVector(W0);
	for(int i=0;i<I.length;++i) {
	    W=getAssociate(W,I[i]);
	    if(W==null) return(false);
	    W=doDynamicsPlus(W,I[i]);
	}
	return(true);
    }



     /**This routine tests if the sequence is feasible for the
        inverse dynamics.*/

    public static boolean testFeasibleInverse(GoldenVector W0,int[] I) {
	GoldenVector G=VerifySupport.getTotalMove(I);
	GoldenVector W=GoldenVector.minus(W0,G);
	W=VerifySupport.fundamentalDomain(W);
	int[] J=Lists.reverse(I);
	return(testFeasible(W,J));
    }


    /**Say that two vectors W1 and W2 are associates if
       W1-W2 lies in (2Z)^3.  This routine tries to
       verify that some associate of W1 lies in the
       correct polyhedron.  If the routine fails, then 
       a null vector is returned.

       We are allowed to move things
       around by associates because points on the boundary
       of the fundamental domain have more than one
       representative.**/

    public static GoldenVector getAssociate(GoldenVector W,int q) {
	for(int i=-4;i<=4;i=i+4) {
	    for(int j=-4;j<=4;j=j+4) {
		for(int k=-4;k<=4;k=k+4) {
		    GoldenVector X=GoldenVector.plus(W,new GoldenVector(i,0,j,0,k,0));
		    if(testInside(X,q)==true) return(X);
		}
	    }
	}
	return(null);
    }

    public static boolean testInside(GoldenVector W,int q) {
	GoldenPolyhedron P=DataPartition.getGoldenPolyhedron(q);
	P=P.scale(new GoldenReal(2,0));
	int[] N=DataPartitionRaw.polyNormal(q);
	int[] A=DataPartitionRaw.faceAnchor(q);
	for(int i=0;i<A.length;++i) {
	    GoldenVector n=DataPartition.getNormal(N[i]);
	    GoldenVector p0=P.V[A[i]];
	    GoldenReal test=GoldenVector.dot(n,GoldenVector.minus(p0,W));
	    if(test.isPositiveOrZero()==false) return(false);
	}
	return(true);
    }

    /**This does one step of the dynamics using golden arithmetic.
       We add twice the vector V2 because everything is scaled
       up by a factor of 2.**/


  public static GoldenVector doDynamicsPlus(GoldenVector V,int type) {
	GoldenVector V2=DataPartition.getMoveGolden(type);
	GoldenVector W=GoldenVector.plus(V,V2);
	W=GoldenVector.plus(W,V2);
	W=VerifySupport.fundamentalDomain(W);
	return(W);
    }

}
