import java.applet.Applet;
import java.awt.*;
import java.awt.event.*;
import java.applet.*;
import java.awt.geom.*;
import java.math.*;

/**This class generates a random tensegrity.
   There are 10 variables.*/

public class Tensegrity {
    
    /**The input to this routine is a random point in
       the unit cube [0,1]^{10}.  The output is 
       parameters more tuned to the tensegrity.
       Here is the meaning of the output parameters

    A0:  slope of horizontal bend image (b)
    A1:  slope of middle bend image (m)
    A2:  slope of top bend image (t)
    A3:  top bend midpoint (x)
    A4:  top bend midpoint (y)
    A5:  middle bend midpoint (x)
    A6:  middle bend midpoint (y)
    A7:  middle bend midpoint (z)
    A8:  middle bend z-slope
    A9:  middle bend pitch

   The variable T restricts the pitches

    */
    
    public static double[] tuneParameters(double[] r,int[] mode,double[] range,double[] line) {
	double[] A=new double[10];

	//bottom-middle-top slopes
	double[] B=slope(r[0],r[1],r[2]);
	A[0]=B[0];
	A[1]=B[1];
	A[2]=B[2];

	//offsets
	A[3]=interpolate(0,1.0/18,r[3]);
	A[4]=interpolate(-1.0/30,1.0/30,r[4]);

	//puts the midpoint of the middle bend on the preferred line

	double slope = Math.tan(line[0]);
	double y_intercept = line[1];
	A[5]=interpolate(-2,1,r[5]);
        A[6]=slope*A[5]+y_intercept;

	//setting the height and vertical angle of the middle bend	
	A[7]=interpolate(-1,1,r[7]);
	A[8]=interpolate(Math.PI/4,-Math.PI/4,r[8]);

	//the  pitch for the middle bend
	A[9]=interpolate(range[0],range[1],r[9]);

	//these are simplifications:  planarity
	if(mode[0]>0) {
	    A[7]=0;
	    A[8]=0;
	}
	//setting the offsets to 0 to get a perfect T
	if(mode[0]==2) {
	    A[3]=0;
	    A[4]=0;
	}

	//getting the middle bend endpoint on the line
	double sign=1;
	if(mode[2]==1) sign=-1;
	double M=Math.sqrt(1+A[1]*A[1]);
	M=M*Math.cos(A[8])/2;
	M=M*sign;
	A[5]=A[5]-M*Math.cos(A[9]);
	A[6]=A[6]-M*Math.sin(A[9]);


	//putting the middle bend endpoint above/below the preferred line
	if(mode[1]==0) A[6]=A[6]+r[6];
	if(mode[1]==2) A[6]=A[6]-r[6];
	
	return A;
    }



    /**this gets the t,m,b coordinates: the slopes of the tensegrity*/

     public static double[] slope(double r0,double r1,double r2) {
	double[] A=new double[3];
	double bmax=(Math.sqrt(27)-Math.sqrt(11))/4;
	A[0]=interpolate(0,bmax,r0);
	A[1]=interpolate(-1,1,r1);
	double y1=(2.0/3)*A[0]-1.0/Math.sqrt(3);
	double y2=(2.0/3)*A[0]-1.0/2;
	double y3=(4.0/3)*A[0]-1.0/Math.sqrt(3);
	double y4=Math.min(y2,y3);
	A[2]=interpolate(y1,y4,r2);
	return A;
    }
    

    public static double interpolate(double a,double b,double r) {
	return (1-r)*a+r*b;
    }
    

    /**This computes the capacity of the tensegrity.
       The vertices of the trapezoid are

       Z[2] = (0,L-b)         Z[1]=(1,R);

       Z[0]=  (0,-b)          (1,0)
    */
    

    public static double capacity(double[] A,int level) {
	double b=A[0];
	Complex[] Z=toTensegrityLeft(A,level);
	double L=Complex.dist(Z[0],Z[2]);
	double R=Complex.dist(new Complex(1,0),Z[1]);
	double S=L+R;
	double cap1=S-Math.sqrt(3);
	return cap1;
    }
    
    public static Vector[] toTensegrity(double[] A) {
	
	Vector[] V=new Vector[6];
	for(int i=0;i<6;++i) V[i]=new Vector();

	//horizontal edge
	double B=Math.sqrt(A[0]*A[0]+1);
	V[0]=new Vector(-B,0,0);

	//middle edge
	double M=Math.sqrt(A[1]*A[1]+1);
        double theta=A[9];
	double phi=A[8];
        double x=(M/2)*Math.cos(theta)*Math.cos(phi);
        double y=(M/2)*Math.sin(theta)*Math.cos(phi);
	double z=(M/2)*Math.sin(phi);
	V[3]=new Vector(A[5]+x,A[6]+y,A[7]+z);
	V[4]=new Vector(A[5]-x,A[6]-y,A[7]-z);

	//vertical edge
	double u=A[3];
	double v=A[4];
	double T=Math.sqrt(A[2]*A[2]+1);
	V[1]=new Vector(u,+T/2+v,0);
        V[2]=new Vector(u,-T/2+v,0);
	return V;
    }



    /**This checks the other relations in the tensegrity*/

    public static boolean isGood(double[] A,int level) {
	Vector[] V=toTensegrity(A);
	Complex[] Z=toTensegrityLeft(A,level);

	double[] t=new double[4];
	
	t[0]=Vector.dist(V[0],V[4])-Complex.dist(Z[0],Z[4]);
	t[1]=Vector.dist(new Vector(0,0,0),V[3])-Complex.dist(new Complex(1,0),Z[3]);
	t[2]=Vector.dist(V[0],V[3])-Complex.dist(Z[0],Z[3]);
	t[3]=Vector.dist(new Vector(0,0,0),V[4])-Complex.dist(new Complex(1,0),Z[4]);

	double min=10;
	double max=-10;
	for(int i=0;i<4;++i) {
	    if(min>t[i]) min=t[i];
	    if(max<t[i]) max=t[i];
	}

	boolean bad=false;
	if(min>.00000000001) bad=true;
	if(max<-.000000001) bad=true;
	if(bad==true) return false;

	t[0]=Vector.dist(V[2],V[4])-Complex.dist(Z[2],Z[4]);
	t[1]=Vector.dist(V[1],V[3])-Complex.dist(Z[1],Z[3]);
	t[2]=Vector.dist(V[1],V[4])-Complex.dist(Z[1],Z[4]);
	t[3]=Vector.dist(V[2],V[3])-Complex.dist(Z[2],Z[3]);


	min=10;
	max=-10;
	for(int i=0;i<4;++i) {
	    if(min>t[i]) min=t[i];
	    if(max<t[i]) max=t[i];
	}

	if(min>.00000000001) bad=true;
	if(max<-.000000001) bad=true;
	if(bad==true) return false;
	return true;
    }


    

    /**takes the spatial tensegrity and projects it into the plane*/
    
    public static Complex[] toTensegrityRight(double[] A) {
	Vector[] V=toTensegrity(A);
	Complex[] Z=new Complex[6];
	for(int i=0;i<6;++i) Z[i]=new Complex(V[i].x[0],V[i].x[1]);
	return Z;
    }
    
    public static double perimeter(double[] A) {
	Vector O=new Vector(0,0,0);
	Vector[] W=toTensegrity(A);
	double peri=Vector.dist(W[0],W[4])+Vector.dist(W[4],W[2])+Vector.dist(W[3],W[1])+Vector.dist(O,W[3]);
	return peri;
    }
    
    public static double perimeterAlt(double[] A) {
	Vector O=new Vector(0,0,0);
	Vector[] W=toTensegrity(A);
	double peri=Vector.dist(W[0],W[1])+Vector.dist(O,W[2]);
	return peri;
    }
    
    

    /**This computes the capacity of the tensegrity.
       The vertices of the trapezoid are

       Z[2] = (0,L1+L2-b)         Z[1]=(1,R1+R2);

       Z[4] = (0,L1-b)            Z[3]=(1,R1);

       Z[0]=  (0,-b)              (1,0)
    */
    
    public static double bValue(double[] A) {
	return A[0];
    }

    
    public static Complex[] toTensegrityLeft(double[] A,int level) {
	Complex[] T=new Complex[5];
	Vector ORIGIN=new Vector(0,0,0);
	double b=A[0];
	double m=A[1];
	double t=A[2];
	
	double diff1=b-m;
	double diff2=m-t;
	Vector[] W=toTensegrity(A);

	//bottom half
	double[] LR1=Trapezoid.main(b,m,Vector.dist(W[0],W[4]),Vector.dist(ORIGIN,W[3]),Vector.dist(W[0],W[3]),Vector.dist(ORIGIN,W[4]),level);
	double L1=LR1[0];
	double R1=LR1[1];

	//top half
	double[] LR2=Trapezoid.main(m,t,Vector.dist(W[2],W[4]),Vector.dist(W[1],W[3]),Vector.dist(W[4],W[1]),Vector.dist(W[2],W[3]),level);
	double L2=LR2[0];
	double R2=LR2[1];
	
	//assemble
	T[0]=new Complex(0,-b);
	T[4]=new Complex(0,L1-b);
	T[2]=new Complex(0,L1+L2-b);

	T[3]=new Complex(1,R1);
	T[1]=new Complex(1,R1+R2);
	return T;	
    }
    
    
}

