import java.applet.Applet;
import java.awt.*;
import java.awt.event.*;
import java.applet.*;
import java.awt.geom.*;
import java.math.*;


public class PolytopeIntersector {


    /**This file numerically computes the intersection of
       two convex polytopes. Here are the steps:

       1.  We identify a 3-face of the polytope with a
           5-vector.  The 5-vector gives a linear functional
           which vanishes on the 3-face.

       2.  We first compute all linear functionals associated
           to all 4-tuples of points in general position.  We
           then eliminate those which take both signs on the
           polytope vertices.  This leaves the boundary 3-faces.
           We adjust the linear functionals to be non-negative
           on the polytope vertices.

       3.  We take the union L of all these boundary 3-faces for  
           both polyhedra.  Taking 4-tuples from U we produce a
           cloud of points.  We then eliminate points on which
           some element of L is not non-negative.  This leaves a
           cloud of points in the intersection.
  
       4.  We use a randomized algorithm (the cheap hull) to
           identify the extreme points in our cloud.  These
           are very likely the vertices of the intersection.

       This algorithm is not guaranteed to work, but it is
       pretty good, and works when we need it to work.*/


    public static Polytope main(Polytope P,Polytope Q) {

	Vector[] L1=linearFunctionalList(P);
	Vector[] L2=linearFunctionalList(Q);
	Vector[] L=merge(L1,L2);
	L=irredundant(L);

	Vector4[] TOTAL=totalList(L);
	Polytope PQ=new Polytope();
	for(int i=0;i<TOTAL.length;++i) {
          PQ.V[i]=TOTAL[i];
	}

	PQ.count=TOTAL.length;
	PQ=cheapHull(PQ);

	System.out.println("done");
	return PQ;

    }


    /**This routine gets the list of positive intersection points
       relative to a list of linear functionals*/

    public static Vector4[] totalList(Vector[] L) {
	Vector4[] LIST=new Vector4[100000];
	int COUNT=0;
	int n=L.length;
	for(int i0=0;i0<n;++i0) {
	for(int i1=i0+1;i1<n;++i1) {
	for(int i2=i1+1;i2<n;++i2) {
	for(int i3=i2+1;i3<n;++i3) {


	    Vector[] TRY={L[i0],L[i1],L[i2],L[i3]};
	    Vector4 VV=intersect(TRY);
	    if(VV!=null) {
	    boolean test=isPositive(L,VV);
	    if(test==true) {
	    if(onList(VV,LIST,COUNT)==false) {
	       LIST[COUNT]=new Vector4(VV);
	       ++COUNT;
	    }}}
	}}}}
	Vector4[] LIST2=new Vector4[COUNT];
	for(int i=0;i<COUNT;++i) LIST2[i]=new Vector4(LIST[i]);
	return LIST2;
    }





    /**This routine checks whether a given point actually belongs
       to the pair of polytopes.  We do this with the list of
       positive functionals*/

    public static boolean isPositive(Vector[] L,Vector4 V) {
	double tol=.00000001;
	for(int i=0;i<L.length;++i) {
	    if(evaluate(L[i],V)<-tol) return false;
	}
	return true;
    }

    /**This finds a point which vanishes on 4 linear
       functionals. In other words, it is the intersection
       of the hyperplanes defined by the linear
       functionals.  In case the linear functionals
       are not independent, it returns null*/


    public static Vector4 intersect(Vector[] L) {
	Vector[] L4=new Vector[4];
	for(int i=0;i<4;++i) {
	    L4[i]=new Vector(L[i].x[0],L[i].x[1],L[i].x[2],L[i].x[3]);
	}
	Vector T=new Vector(-L[0].x[4],-L[1].x[4],-L[2].x[4],-L[3].x[4]);
	Matrix M=Matrix.makeRow(L4);

	double d=M.det();
	if(d*d<.000000000001) return null;

	M=M.inverse();
	Vector V=Matrix.act(M,T);
	Vector4 V4=new Vector4(V.x[0],V.x[1],V.x[2],V.x[3]);
	return V4;
    }



    /**This gets all the linear functionals associated
       to a polytope*/

    public static Vector[] linearFunctionalList(Polytope P) {
	int n=P.count;
	int COUNT=0;
	Vector[] LIST=new Vector[1000];
	for(int i0=0;i0<n;++i0) {
	for(int i1=i0+1;i1<n;++i1) {
	for(int i2=i1+1;i2<n;++i2) {
	for(int i3=i2+1;i3<n;++i3) {
	    Vector4[] A={P.V[i0],P.V[i1],P.V[i2],P.V[i3]};
	    Vector L=linearFunctional(A);
	    L=fixSign(L,P);
	    if(L!=null) {
		if(L.norm()>.00000001) {
		  LIST[COUNT]=new Vector(L);
		  ++COUNT;
		}
	    }

	}}}}
	Vector[] LIST2=new Vector[COUNT];
	for(int i=0;i<COUNT;++i) LIST2[i]=new Vector(LIST[i]);
	return LIST2;
    }


    /**This finds a linear functional that vanishes
       on the 4 points.  The input is a 4-tuple of
       vectors in R^4.  The output is a 5-vector.
       The linear functional is

          L0 x0+L1 x1 + L2 x2 + L3 x3 + L4=0

    */

    public static Vector linearFunctional(Vector4[] A) {
	Vector4[] B=new Vector4[3];
	for(int i=0;i<3;++i) B[i]=Vector4.minus(A[i],A[3]);

	Vector4 W=ortho(B);
	double e=Vector4.dot(W,A[3]);
	double[] temp={W.x[0],W.x[1],W.x[2],W.x[3],-e};
	Vector L=new Vector(temp);
	L=Vector.normalize(L);

	for(int i=0;i<4;++i) {
	    double test=evaluate(L,A[i]);
	    if(test*test>.0000000001) {
		return null;
	    }
	}

	return L;

    }


    /**This routine evaluates a linear functional on 
       a 4-vector*/

    public static double evaluate(Vector L,Vector4 W) {
	double sum=0;
	for(int i=0;i<4;++i) sum=sum+L.x[i]*W.x[i];
	sum=sum+L.x[4];
	return sum;
    }


    /**This routine returns 
       -1 if the linear functional is negative on the polytope, and 
       +1 if it is positive and 
        0 if it is neither.*/


    public static int sign(Vector L,Polytope P) {
	boolean isNeg=false;
	boolean isPos=false;
	double tol=.000000001;
	for(int i=0;i<P.count;++i) {
	    double test=evaluate(L,P.V[i]);
	    if(test<-tol) isNeg=true;
	    if(test>tol) isPos=true;
	}
	if(isNeg==false) return +1;
	if(isPos==false) return -1;
	return 0;
    }



    /**In case the linear functional L is negative on the
       polytope, this returns -L.  If the sign is 0
       a null vector is returned*/

    public static Vector fixSign(Vector L,Polytope P) {
	if(L==null) return null;
	int s=sign(L,P);
	if(s==0) return null;
	if(s==1) return L;
	return Vector.scale(-1,L);
    }



    /**This makes a list of linear functionals irredundant*/

    public static Vector[] irredundant(Vector[] LIST0) {
	int COUNT=0;
	Vector[] LIST1=new Vector[LIST0.length];
	for(int i=0;i<LIST0.length;++i) {
	    if(onList(LIST0[i],LIST1,COUNT)==false) {
		LIST1[COUNT]=new Vector(LIST0[i]);
		++COUNT;
	    }
	}
	Vector[] LIST2=new Vector[COUNT];
	for(int i=0;i<COUNT;++i) LIST2[i]=new Vector(LIST1[i]);
	return LIST2;
    }


    /**This checks if a given linear functional is on a list*/
    public static boolean onList(Vector k,Vector[] LIST,int COUNT) {
	for(int i=0;i<COUNT;++i) {
	    if(Vector.dist(LIST[i],k)<.00000001) return true;
	}
	return false;
    }


    /**This does the same thing for the Vector4 class*/
    public static boolean onList(Vector4 k,Vector4[] LIST,int COUNT) {
	for(int i=0;i<COUNT;++i) {
	    if(Vector4.dist2(LIST[i],k)<.0000000001) return true;
	}
	return false;
    }



    /**This merges a list of linear functionals*/
    public static Vector[] merge(Vector[] L1,Vector[] L2) {
	int count=0;
	Vector[] LIST=new Vector[L1.length+L2.length];
	for(int i=0;i<L1.length;++i) {
	    LIST[count]=new Vector(L1[i]);
	    ++count;
	}

	for(int i=0;i<L2.length;++i) {
	    LIST[count]=new Vector(L2[i]);
	    ++count;
	}
	return LIST;
    }





    /**This finds a vector perpendicular to 3 given ones*/

    public static Vector4 ortho(Vector4[] B) {
	Vector4[] A=new Vector4[4];
	Vector4[] LIST0={};
	A[0]=ortho(B[0],LIST0);
	Vector4[] LIST1={A[0]};
	A[1]=ortho(B[1],LIST1);
	Vector4[] LIST2={A[0],A[1]};
	A[2]=ortho(B[2],LIST2);
	Vector4[] LIST3={A[0],A[1],A[2]};
	Vector4 RAN=new Vector4(Math.random(),Math.random(),Math.random(),1);
	RAN=ortho(RAN,LIST3);
	return RAN;
    }

    /**This does one step of Graham-Schmidt.
       This assumes that the B's are orthonormal already*/

    public static Vector4 ortho(Vector4 a,Vector4[] B) {
	Vector4 C=new Vector4(a);
	for(int i=0;i<B.length;++i) {
	    double t=Vector4.dot(C,B[i]);
	    C=Vector4.minus(C,B[i].scale(t));
	}
	double s=Vector4.dot(C,C);
	s=Math.sqrt(s);
	C=C.scale(1/s);
	return C;
    }

    /**cheap way of taking the convex hull - not guaranteed.*/

    public static Polytope cheapHull(Polytope P) {
	Polytope Q=new Polytope();
	Q.count=0;
	for(int i=0;i<10000;++i) {
	    Vector4 W=randomExtreme(P);
	    boolean test=onList(W,Q);
	    if(test==false) {
		Q.V[Q.count]=new Vector4(W);
		++Q.count;
	    }
	}
	return Q;
    }


    /**gets a random extreme point*/

    public static Vector4 randomExtreme(Polytope P) {
	int index=0;
	Vector4 W=new Vector4();
	for(int i=0;i<4;++i) W.x[i]=1-2*Math.random();
	double max=-10000;
	for(int i=0;i<P.count;++i) {
	    double test=Vector4.dot(P.V[i],W);
	    if(max<test) {
		max=test;
		index=i;
	    }
	}
	return(P.V[index]);
    }

    /**Checks numerically when a vector already is a
       vertex of the polytope.*/

    public static boolean onList(Vector4 W,Polytope P) {
	for(int i=0;i<P.count;++i) {
	    if(Vector4.dist2(W,P.V[i])<.0000001) return true;
	}
	return(false);
    }


}












