import java.math.*;

public class TaylorSeries {

    /**The goal of this class is to get under and over approximations
       to expressions of the form

       Y0 2^{-x/2}+
       Y1 3^{-x/2}+
       Y2 4^{-x/2}+
       Y3 x 2^{-x/2}+
       Y4 x 3^{-x/2}+
       Y5 x 4^{-x/2}.

       on the interval [-2,16].  This is somewhat larger an interval
       than we actually need.  We use 12th order Taylor series
       approximations to each of m^{-2/s} for m=2,3,4, centered
       at even integers -2,,2,4,6,8,10,12,14,16.  The resulting
       series have transcendental coefficients which are rational
       combinations of log(m) for m=2,3,4.  We take specific
       over and under approcimations for these quantities, and
       then make sure to manage the under and over approximation
       correctly.*/



    /**These are the under and over approximations of log m:
       j=0: under approx for log(m)
       j=1: over approx for log(m)
       This is only meant to be used when m=2,3,4.*/

    public static BigRational LOG(int m,int j) {
	long[][] p={{25469,5225,25469},{7050,708784,345197}};
	long[][] q={{36744,4756,18372},{10171,645163,249007}};
	return new BigRational(p[j][m-2],q[j][m-2]);
    }


    public static BigRationalInterval LOG(int m) {
	BigRational L=LOG(m,0);
	BigRational R=LOG(m,1);
	BigRationalInterval Y=new BigRationalInterval(L,R);
	return Y;
    }

    /**(i)th Taylor series coefficient for Power[m,-x/2] 
       centered at x0.  apx =0 or 1 says which 
       log approximation value to get. 
       alt tells whether to get the alternating series or the
       direct series.**/

    public static BigRationalInterval seriesTerm(int m,int x0,int i,int alt) {
	BigRational TWO=new BigRational(2,1);
	BigRational M=new BigRational(m,1);
        BigRationalInterval L=LOG(m);
	BigRational T=new BigRational(1,1);
	if((alt==1)&&(i%2==1)) T=new BigRational(-1,1);
	T=T.divide(BigRational.factorial(i));
	T=T.divide(BigRational.power(TWO,i));
	T=T.divide(BigRational.power(M,x0/2));
	L=BigRationalInterval.powerBasic(L,i);
	L=L.scale(T);
	return L;
    }



    public static Poly1Interval series(long c,int m,int x0,int alt) {
	Poly1Interval P=new Poly1Interval();	
        P.degree=12;
	for(int i=0;i<12;++i) P.C[i]=seriesTerm(m,x0,i,alt);
	BigRational C=new BigRational(c,1);
	for(int i=0;i<12;++i) P.C[i]=P.C[i].scale(C);
	BigRational E1=new BigRational(-1,1);
	BigRational E2=new BigRational(+1,1);	
        BigRational S=BigRational.factorial(12);
        E1=E1.divide(S);
        E2=E2.divide(S);
	P.C[12]=new BigRationalInterval(E1,E2);
	return P;
    }

    public static Poly1Interval combo(long[] Y,int x0,int alt) {
	if(isZero(Y)==true) return new Poly1Interval();
	Poly1Interval P=new Poly1Interval();
	Poly1Interval Q=new Poly1Interval();
	P.degree=12;
	Q.degree=12;
	Q=series(Y[0],2,x0,alt);
	P=P.add(Q);
	Q=series(Y[1],3,x0,alt);
	P=P.add(Q);
	Q=series(Y[2],4,x0,alt);
	P=P.add(Q);
	Q=series(Y[3],2,x0,alt);
	P=addS(P,Q,x0,alt);
	Q=series(Y[4],3,x0,alt);
	P=addS(P,Q,x0,alt);
	Q=series(Y[5],4,x0,alt);
	P=addS(P,Q,x0,alt);
	P.degree=13;
	return P;
    }

    public static Poly1Interval addS(Poly1Interval P,Poly1Interval Q,int x0,int alt) {
	BigRational X0=new BigRational(x0,1);
	Poly1Interval R=P.add(Q.scale(X0));
	Q.shiftUp();
	if(alt==0) R=R.subtract(Q);
	if(alt==1) R=R.add(Q);
	return R;
    }
    /**Here is the main expression. The variable choice refers
       to whether we are over or under approximating:
       0 under, 1 over*/

    public static Poly1 main(int choice,long[] Y,int x0,int alt) {
	if(choice==0) return mainMin(Y,x0,alt);
	return mainMax(Y,x0,alt);
    }

    public static Poly1 mainMin(long[] Y,int x0,int alt) {
	if(isZero(Y)==true) return new Poly1();
	Poly1Interval P=combo(Y,x0,alt);
	Poly1 Q=P.min();
	Q.LEFT=new BigRational(0,1);
	Q.RIGHT=new BigRational(1,1);
	return Q;
    }
    public static Poly1 mainMax(long[] Y,int x0,int alt) {
	if(isZero(Y)==true) return new Poly1();
	Poly1Interval P=combo(Y,x0,alt);
	Poly1 Q=P.max();
	Q.LEFT=new BigRational(0,1);
	Q.RIGHT=new BigRational(1,1);
	return Q;
    }

    public static boolean isZero(long[] Y) {
	for(int i=0;i<5;++i) {
	    if(Y[i]!=0) return false;
	}
	return true;
    }


}

