(* This file corresponds to Lemma L23 in the book.
   This is the combinatorial analysis of the 7th partials of the
   function G_k.  We have this set up exactly as in the lemma.  When
   you load this file into Mathematica it prints out all the
   required bounds.  The last calculation took about half an
   hour in Dec 2023 on my MacBook Pro.*)
   
   
   
(*This is the function that we analyze*)

g[x1_,y1_,x2_,y2_,EXP_]:=(
aa=IST[x1,y1];
bb=IST[x2,y2];
cc=(aa-bb).(aa-bb);
Simplify[Power[4-cc,EXP]])

(*inverse stereographic projection*)
IST[x_, y_] := {(2*x)/(1 + x*x + y*y), (2*y)/(1 + x*x + y*y), 1 - 2/(1 + x*x + y*y)};

(*End of function definition*)




(*This routine lists out all the non-decreasing sequences
of length k using the numbers {1,2,3,4}.  This generates
the multi-indices with respect to which we take partial
derivatives*)

TupleNonDecrease4[k_]:=(
Clear[LIST,TEST,TEST2,LIST2];
LIST=Tuples[Range[4],k];
TEST=Table[NonDecrease[LIST[[j]]],{j,1,Length[LIST]}];
TEST2=Flatten[Position[TEST,0]];
LIST2=Table[LIST[[TEST2[[j]]]],{j,1,Length[TEST2]}];
LIST2)

(*End of sequence list*)





(**This returns the polynomial that
is the Jth partial derivative of the function g[*,EXP] defined above*)

DerPoly[J_,EXP_]:=(
a1=g[x1,y1,x2,y2,EXP];
VAR={x1,y1,x2,y2};
count=1;
Clear[x1,y1,x2,y2];
While[count<=Length[J],{a1=D[a1,VAR[[J[[count]]]]];++count}];
Expand[a1])

(*End of partial derivative definition*)




(*This analyzes the individual terms in the expressions above.
  Each term has the form

        C [a1,b1,a2,b2]/[u,v] where
        
         [a1,b1,a1,b2 = x1^a1 y1^b1 x2^a2 y2^b2  and

         [u,v] = (1+x1^2 + y1^2)^u (1+x2^2 + y2^2)^v  and
          
         a1+b1 <= 2u  and
          
         a2+b2 <= 2v
          
         The bound for such a term is |C| times (1/2)^s where
          
         s=Min[a1,b1] + Min[a2,b2] *)
        

ProcessTerm[x_]:=(
nx=Numerator[x];
dx=Denominator[x];
xx1=Exponent[nx,x1];
yy1=Exponent[nx,y1];
xx2=Exponent[nx,x2];
yy2=Exponent[nx,y2];
ss=Min[xx1,yy1]+Min[xx2,yy2];
coeff=Abs[nx//.{x1->1,y1->1,x2->1,y2->1}];
Power[1/2,ss] coeff)

(*End of term processing routine*)




(*This takes the given partial derivative and processes all
the terms according to the routine above and then adds the
result.  This gives a global upper bound to the value of the
partial derivative on R^4*)

             
GetBound[J_,EXP_]:=(
a2=DerPoly[J,EXP];
a3=Table[ProcessTerm[a2[[j]]],{j,1,Length[a2]}];
a4=Sum[a3[[j]],{j,1,Length[a3]}];
a4);

(*End of bound on partial derivative*)




(*This uses the previous routine GetBound for all possible
    partial derivatives of order DIFF and takes the max*)

MaxCombinatorics[DIFF_,EXP_]:=(
LIST=TupleNonDecrease4[DIFF];
BOUND=Table[GetBound[LIST[[j]],EXP],{j,1,Length[LIST]}];
Max[BOUND])

(* End of our max combinatorial bound routine*)






(*This routine prints out the MaxCombinatorics bound scaled
by the Taylor series term that is present in the proof of 
Lemma A11224.  

We set scale =1 when EXP=1,2,3,4,5,6 and
  
We set scale=1/32 when EXP=10*)
   
PrintBoundTaylor[DIFF_,EXP_,scale_]:=(
Print["derivative ",DIFF," exponent ",EXP];
COM=MaxCombinatorics[DIFF,EXP];
TAY=Power[7 Power[2,-18],DIFF-3]/(DIFF-3)!;
Print[4 COM TAY scale];
Print[4 COM TAY scale//N];
)

(*end of main print routine*)




(* For the purpose of experimentation, this routine
takes a shortcut and just computes the derivative
with respect to (1,...,1).  This is the conjectured
optimal one, and quickly gives the same answer
as the much slower routine PrintBoundTaylor*)

PrintQuickTaylor[DIFF_,EXP_,scale_]:=(
Print["derivative ",DIFF," exponent ",EXP];
ONE=Table[1,{DIFF}];
COM=GetBound[ONE,EXP];
TAY=Power[7 Power[2,-18],DIFF-3]/(DIFF-3)!;
Print[4 COM TAY scale];
Print[4 COM TAY scale//N];
)

(* End of quick experimentla routine*)



(*And here is the printout*)

PrintBoundTaylor[7,1,1]
PrintBoundTaylor[7,2,1]
PrintBoundTaylor[7,3,1]
PrintBoundTaylor[7,4,1]
PrintBoundTaylor[7,5,1]
PrintBoundTaylor[7,6,1]
PrintBoundTaylor[7,10,1/32]

