
import java.awt.*;
import java.awt.event.*;
import java.applet.*;
import java.awt.geom.*;
import java.math.*;


/**This class defines the spherical measures which go into
   the energy theorem.  These are described in Part 3 of the
   monograph*/

public class SphericalMeasures {


    /**Gets the vertices of a box and then does inverse 
       stereographic projection.
       The boxes of type 2 corresponds to 
       the infinite point in the plane, which
       in turn corresponds to the point 
       (0,0,1) on the unit sphere.  **/

    public static Vector[] sphereVectors(Box B) {

	if(B.type==2) {
	    Vector[] V={new Vector(0,0,1)};
	    return(V);
	}

	Complex[] Z=B.toVertices();
	Vector[] V=new Vector[Z.length];
	for(int i=0;i<Z.length;++i) V[i]=Vector.inverseStereo(Z[i]);
	return(V);
    }


    /**center of mass of the spherical vertices*/

     public static Vector centerOfMassSpherical(Box B) {
	Vector[] V=sphereVectors(B);
	if(V.length==1) return V[0];
	Vector W=new Vector(0,0,0);
	for(int i=0;i<V.length;++i) W=Vector.plus(W,V[i]);
	W=W.scale(1.0/V.length);
	return W;
    }


    /**This computes the square of the  max diameter of a side of
       the convex hull of the box vertices.*/

    public static double sideDiameterMaxSquared(Box B) {
	Vector[] V=sphereVectors(B);
	if(V.length==1) return(0);
	double r=0;
	for(int i=0;i<V.length;++i) {
	    int j=(i+1)%V.length;
	    double test=Vector.distSquared(V[i],V[j]);
	    if(r<test) {
                r=test;
	    }
	}
	return r;
    }


    /**This is the square of the diameter of the convex hull of the
       vertices of the box. It is used in our energy estimate*/

    public static double hullDiameterSquared(Box B) {
	Vector[] V=sphereVectors(B);
	if(V.length==1) return(0);
	double max=0;
	for(int i=0;i<V.length;++i) {
	    for(int j=i+1;j<V.length;++j) {
		double test=Vector.distSquared(V[i],V[j]);
		if(max<test) max=test;
	    }
	}
	return max;
    }

    /**This is the diameter of the image of the circle on S2
       containing the box*/

    public static double crudeMeasureSquared(Box B) {
	double R2=B.z.x*B.z.x+B.z.y*B.z.y;   //radius of center, squared
	double r2=Math.pow(0.5,2*B.k+1);    //diagonal radius, squared
	double a=16*r2;
        double b=1+ 2*r2 + 2*R2+(R2-r2)*(R2-r2);
	double diam2=a/b;
	return(diam2);
    }

    /**This is the hull separation constant*/

    public static double hullSepConstant(Box B) {
	if(B.type==2) return 0;	
        double crude=crudeMeasureSquared(B);
	double c2=circleBound2(crude);
	if(B.type==1) return c2;
	double side=sideDiameterMaxSquared(B);
	double c1=circleBound1(side);
	double c=Math.max(c1,c2);
	return(c1);
    }


     public static double circleBound1(double d) {
	 double x=d/4 + (d*d)/2;
	 return x;
     }


     public static double circleBound2(double d) {
	 double x=d/8 + (d*d)/16;
	 return x;
     }




    public static double dotMinLower(Box B1,Box B2) {
	Vector[] V1=sphereVectors(B1);
	Vector[] V2=sphereVectors(B2);
	double min=1;
	for(int i=0;i<V1.length;++i) {
	    for(int j=0;j<V2.length;++j) {
		double test=Vector.dot(V1[i],V2[j]);
		if(min>test) min=test;
	    }
	}
        if(B1.type==2) return min;
	if(B2.type==2) return min;
	double d1=hullSepConstant(B1);
	double d2=hullSepConstant(B2);
	min=min-d1-d2-d1*d2;
	return min;
    }

    public static double dotMaxUpper(Box B1,Box B2) {
	Vector[] V1=sphereVectors(B1);
	Vector[] V2=sphereVectors(B2);
	double max=-1;
	for(int i=0;i<V1.length;++i) {
	    for(int j=0;j<V2.length;++j) {
		double test=Vector.dot(V1[i],V2[j]);
		if(max<test) max=test;
	    }
	}
	if(B1.type==2) return max;
	if(B2.type==2) return max;
	double d1=hullSepConstant(B1);
	double d2=hullSepConstant(B2);
	max=max+d1+d2+d1*d2;
	return max;
    }

}
