import java.applet.Applet;
import java.awt.event.*;
import java.awt.*;
import java.math.*;

/**This program contains the basic data structure for our rigorous
proof in the Penrose tile example. The integer sequence x0,x1,x2,x3
represents the complex number

(x0+x1 phi)/2 + I (x2/2 + x3 phi/2)

*/


    /*The basic constructor routines*/

public class IntegerComplex {
    int[] x=new int[4];
    
    public IntegerComplex() {
	x[0]=0;
	x[1]=0;
	x[2]=0;
	x[3]=0;
    } 
    
    public IntegerComplex(int a0,int a1,int a2,int a3) {
	x[0]=a0;
	x[1]=a1;
	x[2]=a2;
	x[3]=a3;
    } 

    public IntegerComplex(IntegerComplex z) {
	x[0]=z.x[0];
	x[1]=z.x[1];
	x[2]=z.x[2];
	x[3]=z.x[3];
    }

    
    public void print() {
       System.out.println(x[0]+" "+x[1]+"     "+x[2]+" "+x[3]);
    }

    public void print2() {
	Complex z=this.toComplex();
	z.print();
    }



    /**converts to a complex number*/

    public Complex toComplex() {
	double PHI=(1+Math.sqrt(5))/2;
	Complex y=new Complex(x[0]*.5+x[1]*.5*PHI,x[2]*.5+x[3]*PHI*.5);
	return(y);
    }

    /**These are the basic arithmetic operations*/

    /**Addition*/

    public static IntegerComplex plus(IntegerComplex z1,IntegerComplex z2) {
	IntegerComplex w=new IntegerComplex();
	for(int i=0;i<4;++i) w.x[i]=z1.x[i]+z2.x[i];
	return(w);
    }

    /**subtraction*/

    public static IntegerComplex minus(IntegerComplex z1,IntegerComplex z2) {
	IntegerComplex w=new IntegerComplex();
	for(int i=0;i<4;++i) w.x[i]=z1.x[i]-z2.x[i];
	return(w);
    }

    /**conjugation*/

    public static IntegerComplex conjugate(IntegerComplex z) {
	IntegerComplex w=new IntegerComplex();
	w.x[0]=z.x[0];
	w.x[1]=z.x[1];
	w.x[2]=-z.x[2];
	w.x[3]=-z.x[3];
	return(w);
    }


    /*This routine takes care of multiplying elements of Z[phi].  It is
      afterwards incorporated into our routine for multiplication*/

    public static int[] times(int[] x,int[] y) {
	int[] z=new int[2];
	z[0]=x[0]*y[0]+x[1]*y[1];
	z[1]=x[1]*y[0]+x[0]*y[1]+x[1]*y[1];
	return(z);
    }

   /**multiplication:  given a and b, the return is 2*a*b */

     public static IntegerComplex twiceTimes(IntegerComplex z1,IntegerComplex z2) {
	 int[] z11={z1.x[0],z1.x[1]};
	 int[] z12={z1.x[2],z1.x[3]};
	 int[] z21={z2.x[0],z2.x[1]};
	 int[] z22={z2.x[2],z2.x[3]};
	 int[] rw1=times(z11,z21);
	 int[] rw2=times(z12,z22);
	 int[] rw={rw1[0]-rw2[0],rw1[1]-rw2[1]};
	 int[] iw1=times(z11,z22);
	 int[] iw2=times(z21,z12);
	 int[] iw={iw1[0]+iw2[0],iw1[1]+iw2[1]};
	 IntegerComplex w=new IntegerComplex(rw[0],rw[1],iw[0],iw[1]);
	 return(w);
     }


    /*requires all entries even*/

    public IntegerComplex carefulHalf() {
	IntegerComplex q=new IntegerComplex(x[0]/2,x[1]/2,x[2]/2,x[3]/2);
	return(q);
    }


    /**These are the interpolation routines, designed to choose some point
       that lies between two other points*/

    public static IntegerComplex goldenInterpolate(IntegerComplex z1,IntegerComplex z2) {
	IntegerComplex s1=new IntegerComplex(-2,2,0,0);
	IntegerComplex s2=new IntegerComplex(4,-2,0,0);
	IntegerComplex w1=twiceTimes(z1,s1);  //all entries even
	IntegerComplex w2=twiceTimes(z2,s2);  //all entries even
	w1=w1.carefulHalf();
	w2=w2.carefulHalf();
	IntegerComplex w=plus(w1,w2);
	return(w);
    }

    public static int[] goldenInterpolate(int[] x1,int[] x2) {
	IntegerComplex y1=new IntegerComplex(x1[0],x1[1],0,0);
	IntegerComplex y2=new IntegerComplex(x2[0],x2[1],0,0);
	IntegerComplex y=goldenInterpolate(y1,y2);
	int[] a={y.x[0],y.x[1]};
	return(a);
    }




    public static IntegerPolyWedge goldenInterpolate(IntegerPolyWedge P) {
	IntegerPolyWedge Q=new IntegerPolyWedge();
	Q.count=P.count;
	for(int i=0;i<P.count;++i) {
	    int i1=i;
	    int i2=(i+1)%P.count;
	    IntegerComplex z1=new IntegerComplex(P.z[i1][0],P.z[i1][1],P.z[i1][2],P.z[i1][3]);
	    IntegerComplex z2=new IntegerComplex(P.z[i2][0],P.z[i2][1],P.z[i2][2],P.z[i2][3]);
	    IntegerComplex w=goldenInterpolate(z1,z2);
	    Q.z[i][0]=w.x[0];
	    Q.z[i][1]=w.x[1];
	    Q.z[i][2]=w.x[2];
	    Q.z[i][3]=w.x[3];
	}
	return(Q);
    }







    /**this routine returns a 
     -1 if the number a + b phi is negative
      0 if the number a+  b phi is zero
     +1 if the number a + b phi is positive
     99 if the test does not determine the sign.
     The test might be inconclusive because we use integer multiplications
     to determine this.  The essential idea is to approximate phi by
     consecutive Fibonacci fractions, one above and one below.

    */


    public static int sign(int[] x) {

	if((x[0]==0)&&(x[1]==0)) return(0);

	BigInteger A1=new BigInteger("12586269025");    //Fibo(50)
	BigInteger A2=new BigInteger("20365011074");    //Fibo(51)
	BigInteger A3=new BigInteger("32951280099");    //Fibo(52)
	Integer x0=new Integer(x[0]);
	Integer x1=new Integer(x[1]);
	BigInteger X0=new BigInteger(x0.toString());
	BigInteger X1=new BigInteger(x1.toString());

	BigInteger S1=A1.multiply(X0);
	S1=S1.add(A2.multiply(X1));
	BigInteger S2=A2.multiply(X0);
	S2=S2.add(A3.multiply(X1));
	BigInteger ZERO=new BigInteger("0");

	int test1=S1.compareTo(ZERO);
	int test2=S2.compareTo(ZERO);
	if((test1<0)&&(test2<0)) return(-1);
	if((test1>0)&&(test2>0)) return(1);
	return(0);
    }


    public static int checkSmall(int[] a) {
	if((a[0]==1)&&(a[1]==0)) return(1);
	if((a[0]==-1)&&(a[1]==0)) return(1);
	int[] x1={a[0]+1,a[1]};
	int[] x2={-a[0]+1,-a[1]};
	if(sign(x1)!=1) return(0);
	if(sign(x2)!=1) return(0);
	return(1);
    }

    public static int checkSmall8(int[] a) {
	if((a[0]==8)&&(a[1]==0)) return(1);
	if((a[0]==-8)&&(a[1]==0)) return(1);
	int[] x1={a[0]+8,a[1]};
	int[] x2={-a[0]+8,-a[1]};
	if(sign(x1)!=1) return(0);
	if(sign(x2)!=1) return(0);
	return(1);
    }





        public static int signArea(IntegerComplex z1,IntegerComplex z2,IntegerComplex z3) {
	    IntegerComplex w1=minus(z1,z3);
	    IntegerComplex w2=minus(z2,z3);
	    IntegerComplex w=twiceTimes(w1,conjugate(w2));
	    int[] a={w.x[2],w.x[3]};
	    return(sign(a));
	}


    /**this routine returns a 
      1 if the point is inside the closed polygon.
      0 if the point is outside the closed polygon.
      99 if the computation goes bad  */
    

    public static int isClosedContained(IntegerComplex z,IntegerPolyWedge P) {
	int pos=0;
	int neg=0;
	for(int i=0;i<P.count;++i) {
	    int i1=i;
	    int i2=(i+1)%P.count;
	    IntegerComplex w1=new IntegerComplex(P.z[i1][0],P.z[i1][1],P.z[i1][2],P.z[i1][3]);
	    IntegerComplex w2=new IntegerComplex(P.z[i2][0],P.z[i2][1],P.z[i2][2],P.z[i2][3]);
	    int s=signArea(z,w1,w2);
	    if(s==-1) neg=1;
	    if(s== 1) pos=1;
	    if(s==99) {
		   return(99);
	    }
	}
	if((neg==1)&&(pos==1)) return(0);
	return(1);
    }



    /**this routine returns a 
      1 if the point is inside the open polygon.
      0 if the point is outside the open polygon.
      99 if the computation goes bad  */
    

    public static int isOpenContained(IntegerComplex z,IntegerPolyWedge P) {
	int pos=0;
	int neg=0;
	for(int i=0;i<P.count;++i) {
	    int i1=i;
	    int i2=(i+1)%P.count;
	    IntegerComplex w1=new IntegerComplex(P.z[i1][0],P.z[i1][1],P.z[i1][2],P.z[i1][3]);
	    IntegerComplex w2=new IntegerComplex(P.z[i2][0],P.z[i2][1],P.z[i2][2],P.z[i2][3]);
	    int s=signArea(z,w1,w2);
	    if(s==-1) neg=1;
	    if(s== 1) pos=1;
	    if(s==0) return(0);
	    if(s==99) {
		   return(99);
	    }
	}
	if((neg==1)&&(pos==1)) return(0);
	return(1);
    }






  /**this routine returns a 
      1 if the point is inside the closed polygon or a nerby lattice translate
      0 if the point is outside the closed polygon and all nearby lattice translates 
      99 if the computation goes bad*/


    public static int isLatticeClosedContained(IntegerComplex z,IntegerPolyWedge P) {
	for(int a=-6;a<=6;++a) {
	    for(int b=-6;b<=6;++b) {
		IntegerComplex zz=moveLattice(z,a,b);
		Complex temp=zz.toComplex();
		if(isClosedContained(zz,P)==1) return(1);
		if(isClosedContained(zz,P)==99) return(99);
	    }
	}
	return(0);
    }


  /**this routine returns a 
      1 if the point is inside the open polygon or a nerby lattice translate
      0 if the point is outside the open polygon and all nearby lattice translates 
      99 if the computation goes bad*/

    public static int isLatticeOpenContained(IntegerComplex z,IntegerPolyWedge P) {
	for(int a=-6;a<=6;++a) {
	    for(int b=-6;b<=6;++b) {
		IntegerComplex zz=moveLattice(z,a,b);
		Complex temp=zz.toComplex();
		if(isOpenContained(zz,P)==1) return(1);
		if(isOpenContained(zz,P)==99) return(99);
	    }
	}
	return(0);
    }


    /**This routine effects the action of the lattice on our points*/

    public static IntegerComplex moveLattice(IntegerComplex zz, int a,int b) {
	IntegerComplex y1=new IntegerComplex(0,0,2*a,0);
	IntegerComplex y2=new IntegerComplex(2*b,0,0,0);
	IntegerComplex z=new IntegerComplex(zz.x[0],zz.x[1],zz.x[2],zz.x[3]);
	z=plus(z,y1);
	z=plus(z,y2);
	return(z);
    }


    /*This routine checks if an IntegerComplex is zero mod the lattice.
      It only tests nearby points. 1 means yes*/

    public static int isZeroLattice(IntegerComplex z) {
	IntegerComplex Z=new IntegerComplex(z.x[0],z.x[1],z.x[2],z.x[3]);
	for(int i=-10;i<=10;++i) {
	    for(int j=-10;j<=10;++j) {
		Z=IntegerComplex.moveLattice(z,i,j);
		if((Z.x[0]==0)&&(Z.x[1]==0)&&(Z.x[2]==0)&&(Z.x[3]==0)) return(1);
	    }
	}
	return(0);
    }

    /*This routine checks if two IntegerComplexes agree mod the lattice*/

    public static int agreeLattice(IntegerComplex z1,IntegerComplex z2) {
	IntegerComplex z=minus(z1,z2);
	return(isZeroLattice(z));
    }



    /**These are the basic shrinking routines.  In the paper these are the special similarities*/

    public static IntegerComplex shrink(IntegerComplex z) {

	IntegerComplex s1=new IntegerComplex(6,-4,0,0);
	IntegerComplex s2=new IntegerComplex(-3,2,2,-1);
	IntegerComplex w=minus(z,s2);
	w=twiceTimes(w,s1);
	w=w.carefulHalf();
	w=plus(w,s2);
	return(w);
    }

    /**These routines implement the drawing of the arithmetic graph*/

    /*This is the map psi from the paper, written in IntegerComplex coordinates.*/




    public static double toDouble(int[] a) {
	double PHI=(1+Math.sqrt(5))/2;
	return(.5*a[0]+.5*a[1]*PHI);
    }

    /*This routine uses floating point arithmetic to guess the right value
      for the integer nearest the point of interest. However, it doesn't use
      any rounding-off properties of floating point arithmetic.*/

    public static int[] dec(int[] a) {
	double d=toDouble(a);
	int n=(int)(Math.floor(d+.5));
	int[] b={a[0]-2*n,a[1]};
	int test=checkSmall(b);
	if(test==0) {
	    System.out.println("fail "+b[0]+" "+b[1]);
	}
	return(b);
    }

    public static IntegerComplex psi(int a,int b) {
	int x=4-12*a+4*b;
	int y=8*a-2;
	int[] c1={(-x+y)/2,x/2};
	int[] c2={x/2,y/2};
	c1=dec(c1);
	c2=dec(c2);
	IntegerComplex c=new IntegerComplex(c1[0],c1[1],c2[0],c2[1]);
	return(c);
    }




    public static int autoClassify(PenrosePartition PE,int a,int b) {
	IntegerComplex z=psi(a,b);
	for(int i=1;i<=23;++i) {
	    if(isLatticeClosedContained(z,PE.IPOLY[i])==1) return(i);
	}
	return(0);
    }


    /**This routine classifies any point in the arithmetic graph
       according to which gene it is the center of.  We only apply
       this routine to the component of the arithmetic graph
       containing the origin.*/

    public static int fineClassify(PenroseProofData PD,int a,int b) {
      IntegerComplex z=psi(a,b);
      for(int i=0;i<75;++i) {
        if(isLatticeOpenContained(z,PD.Y[i])==1) return(i);
      }
      return(-1);
    }

    public static int classifyShadowType(PenroseProofData PD,int a,int b) {
	int k=fineClassify(PD,a,b);
	int[] X={PD.loc2[k][2],PD.loc2[k][3]}; 
	return(fineClassify(PD,X[0],X[1]));
    }

}

