/******************************************************
                    interval_arithmetic.c

This file does the interval arithmetic
as specified in the paper 

There are a finite number of doubles, and these inherit
a linear order from their inclusion into the reals.
Each double has a successor and a predecessor.  The
successor is the minimally larger double and the
predecessor is the minimally smaller double.
This series of routines finds the successor and
predecessor to a double.  To do this, we
convert a double to an exact--which is a pair of
integers--and then to add or subtract 1 from the
exact.   We do this by casting a pointer to a double
as an integer*/
/****************************************************/


/****************************************/
/*Conversion between doubles and exacts*/
/****************************************/

double exact_to_double(E)
exact E;
{
double *b1, a;
int *b2;
b1=&a;
b2=((int*) b1);
*b2=E.a;
++b2;
*b2=E.b;
return(a);
}

exact double_to_exact(a)
double a;
{
exact E;
double *b1;
int *b2;
b1=&a;
b2=((int*) b1);
E.a=*b2;
++b2;
E.b=*b2;
return(E);
}



/***************************************************/
/***********rounding out****************************/
/***************************************************/


exact exact_successor(E)
exact E;
{
exact F;
F.b=E.b+1;
if(F.b==0) F.a=E.a+1;
if(F.b!=0) F.a=E.a;
return(F);
}


exact exact_predecessor(E)
exact E;
{
exact F;
F.b=E.b-1;
if(E.b==0) F.a=E.a-1;
if(E.b!=0) F.a=E.a;
return(F);
}


double positive_successor(x)
double x;
{
double y,s;
exact E;

E=double_to_exact(x);
E=exact_successor(E);
y=exact_to_double(E);
return(y);
}

double positive_predecessor(x)
double x;
{
double y;
exact E;
E=double_to_exact(x);
E=exact_predecessor(E);
y=exact_to_double(E);
return(y);
}

/*to avoid underflow errors, we will
take (+-) 2^(-512) as the successor and
predecessor of 0.*/

double small_number()
{
  double d;
  exact E;
  E.a=535822336;
  E.b=0;
  d=exact_to_double(E);
  return(d);
}


double successor(x)
double x;
{
  double y,s;
  s=small_number();
  if((x>=0.0)&&(x<s))   y=positive_successor(s);
  if((x<0.0)&&(-x<s))  y=positive_successor(s);
  if(x>s)  y=positive_successor(x);
  if(x<-s) y=-positive_predecessor(-x);
  return(y);
}


double predecessor(x)
double x;
{
  double y,s;
  s=small_number();
  if((x>=0.0)&&(x<s))   y=-positive_successor(s);
  if((x<0.0)&&(-x<s))  y=-positive_successor(s);
  if(x>=s)  y=positive_predecessor(x);
  if(x<-s) y=-positive_successor(-x);
  return(y);
}


/*Here the our rounding routine.  It takes an
interval and rounds it outward in the optimal
way, to produce the minimally larger interval*/

interval round_out(I)
interval I;
{
  
interval J;
J.l=predecessor(I.l);
J.r=successor(I.r);
J.bad=I.bad;
return(I);
}


/***********************************************/
/*************conversion to an interval******/ 
/***********************************************/


interval double_to_interval(d)
double d;
{
interval I;
 I.l=d-0.00000000000001;
 I.r=d+0.00000000000001;
I.bad=0;
return(I);
}




/************************************************/
/************interval arithmetic******************/
/************************************************/

interval max(I,J)
interval I,J;
{
interval K;
K.r=I.r;
if(I.r<J.r) K.r=J.r;
K.l=K.r;
return(K);
}

interval plus(I,J)
interval I,J;
{
interval K;
K.l=I.l+J.l;
K.r=I.r+J.r;
K=round_out(K);
return(K);
}

interval minus(I,J)
interval I,J;
{
interval K;
K.l=I.l-J.r;
K.r=I.r-J.l;
K=round_out(K); 
return(K);

}

// MODIFICATION: Changed to avoid conflict with standard library.
interval user_times(I, J)
interval I,J;
{
double ll,lr,rl,rr,l,r;
interval K;
ll=I.l*J.l;
lr=I.l*J.r;
rl=I.r*J.l;
rr=I.r*J.r;
K.l=ll;
if(K.l>lr) K.l=lr;
if(K.l>rl) K.l=rl;
if(K.l>rr) K.l=rr;
K.r=ll;
if(K.r<lr) K.r=lr;
if(K.r<rl) K.r=rl;
if(K.r<rr) K.r=rr;
K=round_out(K);
return(K);
}


interval divide(I,J)
interval I,J;
{
double ll,lr,rl,rr,l,r;
interval K;
ll=I.l/J.l;
lr=I.l/J.r;
rl=I.r/J.l;
rr=I.r/J.r;
K.l=ll;
if(K.l>lr) K.l=lr;
if(K.l>rl) K.l=rl;
if(K.l>rr) K.l=rr;
K.r=ll;
if(K.r<lr) K.r=lr;
if(K.r<rl) K.r=rl;
if(K.r<rr) K.r=rr;
K=round_out(K);
if((J.l<=0.0)&&(J.r>=0.0))  /*unequal signs problem*/
   {
   K.l=0.0;
   K.r=0.0;
   K.bad=1;
   }
return(K);
}


/*We will only use the square root function when we know
in advance that both endpoints of the interval are
positive.  If the round-off errors cause an endpoint to be
negative the program will have an error*/

interval sq_root(I)
interval I;
{
interval J;
double sqrt();
if(I.l<0.0) J.l=0.0;
if(I.l>=0.0) J.l=sqrt(I.l);
if(I.r<0.0) 
   {
   printf("error: sq_root of negative number\n");
   }
J.r=sqrt(I.r);
J=round_out(J);
if(J.l<0.0) J.l=0.0;
J.bad=I.bad;
return(J);
}


/*used as a maximum function*/

interval absorb(I,J)
     interval I,J;
{
  interval K;
  if(I.l<=J.l) K.l=I.l;
  if(J.l<=I.l) K.l=J.l;
  if(I.r>=J.r) K.r=I.r;
  if(J.r>=I.r) K.r=J.r;
  return(K);
}




/*the name says it all*/
interval ZERO()
{
interval z;
z.l=0.0;
z.r=0.0;
return(z);
}



/*near equality*/

int equal(I1,I2)
     interval I1,I2;
{
  if(I1.r<I2.l) return(0);
  if(I1.l>I2.r) return(0);
  return(1);
}

int cx_equal(z1,z2)
     interval_complex z1,z2;
{
  if(equal(z1.x,z2.x)==0) return(0);
  if(equal(z1.y,z2.y)==0) return(0);
  return(1);
}


int vec_equal(v1,v2)
     interval_vector v1,v2;
{
  if(cx_equal(v1.a,v2.a)==0) return(0); 
  if(cx_equal(v1.b,v2.b)==0) return(0);
  if(cx_equal(v1.c,v2.c)==0) return(0);
  return(1);
}
