#include <stdio.h>
#include <math.h>
#define Pi 3.141592653589
#define RR 0



/***************************************
This program is part of the DENTED TORI package,
written by Richard Evan Schwartz.  It is used
bu the Tcl program surface.tcl, and does not
successfully run on its own.  surface.tcl
automatically does the compiling.
********************************************/

/*********************************************
          APPROXIMATE FILE DIRECTORY

Basic Structures---------------------- 20-80
Algebra--------------------------------80-440
Matrices-------------------------------440-640
Line Drawing Functions-----------------640-770
Surface Pieces-------------------------770-950
Group Action on Pieces-----------------950-980
Conversion to TCL Types----------------980-1010
TCL Interface--------------------------1010-1050
Main Routine---------------------------1050-1150
**************************************************/

/* Basic Structures */

typedef struct 
{char c[6];}
color;

typedef struct 
{double x,y;}
complex;

typedef struct 
{double x,y,z;}
rvector;

typedef struct 
{complex a,b,c;}
vector;

typedef struct 
{vector a,b,c;}
matrix;
 
typedef struct 
{double x,y;}
point;

typedef struct 
{int l;
vector v[210];}
line;

typedef struct
{int l;
line x[70];}
piece;

typedef struct 
{int l;
int w[30];} 
element;

typedef struct 
{int l;
point p[210];}
curve;

typedef struct 
{point p;
double r;
double t1,t2;}
arc;

typedef struct 
{point p,q;
double a1,a2;}
tcl_arc;

typedef struct
{double t;
complex p[5];
 double w[5];}
deformation;


/************basic algebra for the complex numbers*************/

complex plus(p,q)
complex p,q;
{
complex r;
r.x=p.x+q.x;
r.y=p.y+q.y;
return(r);
}

complex minus(p,q)
complex p,q;
{
complex r;
r.x=p.x-q.x;
r.y=p.y-q.y;
return(r);
}

complex times(p,q)
complex p,q;
{
complex r;
r.x=p.x*q.x-p.y*q.y;
r.y=p.x*q.y+p.y*q.x;
return(r);
}

complex inverse(z)
complex z;
{
complex w;
w.x= z.x/(z.x*z.x+z.y*z.y);
w.y=-z.y/(z.x*z.x+z.y*z.y);
return(w);
}

complex conjugate(z)
complex z;
{
complex w;
w.x= z.x;
w.y=-z.y;
return(w);
}

vector conj(w)
vector w;
{
vector cw;
cw.a=conjugate(w.a);
cw.b=conjugate(w.b);
cw.c=conjugate(w.c);
return(cw);
}

complex divide(p,q)
complex p,q;
{
complex r;
r=times(p,inverse(q));
return(r);
}

/*This version of ArcTan takes values in [0,2 Pi]*/

double new_atan(x,y)
double x,y;
{
double p;
double atan2();
p=atan2(y,x);
if (p<0) p=p+2*Pi;
return(p);
}


/*********************************************************
This series of functions computes the projection of a vector
on the three sphere--from our point of view a
Lorenz-Null vector onto the two sphere via the Hopf projection
*************************************************************/

point hopf(v)
vector v;
{
complex ca;
point p;
ca=divide(v.a,v.b);
ca=divide(v.b,v.c);
p.x=ca.x;
p.y=ca.y;
return(p);
}

curve line_to_hopf(v)
line v;
{
curve p;
int i;
p.l=v.l;
for(i=1;i<=v.l;++i)
p.p[i]=hopf(v.v[i]);
return(p);
}

curve line_to_hopf_tcl(v)
line v;
{
curve p;
int i;
p.l=v.l;
for(i=1;i<=v.l;++i)
   {
   p.p[i]=hopf(v.v[i]);
   p.p[i].x=5+5*p.p[i].x;
   p.p[i].y=5-5*p.p[i].y;
   }
return(p);
}


/*********************************************************
This series of functions computes the projection of a vector
on the three sphere--from our point of view a
Lorenz-Null vector onto the xy-plane in Heisenberg space.
*************************************************************/


point plan(v)
vector v;
{
complex cb;
point p;
cb=divide(v.b,v.c);
p.x=cb.x;
p.y=cb.y;
return(p);
}

curve line_to_plan(v)
line v;
{
curve p;
int i;
p.l=v.l;
for(i=1;i<=v.l;++i)
p.p[i]=plan(v.v[i]);
return(p);
}

curve line_to_plan_tcl(v)
line v;
{
curve p;
int i;
p.l=v.l;
for(i=1;i<=v.l;++i)
   {
   p.p[i]=plan(v.v[i]);
   p.p[i].x=5+5*p.p[i].x;
   p.p[i].y=5-5*p.p[i].y;
   }
return(p);
}


/*********************************************************
This series of functions computes the projection of a vector
on the three sphere--from our point of view a
Lorenz-Null vector onto cylinder at infinity about the
z axis, in Heisenberg space.
*************************************************************/


point elev(v)   /* A Single Vector */ 
vector v;
{
complex ca, cb;
point p;
ca=divide(v.a,v.c);
cb=divide(v.b,v.c);
p.x=new_atan(cb.x,cb.y);
p.y=3.0*ca.y;
return(p);
}

curve line_to_elev(v)  /*A list of vectors*/
line v;
{
curve p;
int i;
p.l=v.l;
for(i=1;i<=v.l;++i)
p.p[i]=elev(v.v[i]);
return(p);
}

curve line_to_elev_tcl(v)  /*Tcl compatible version*/
line v;
{
curve p;
int i;
p.l=v.l;
for(i=1;i<=v.l;++i)
  {
  p.p[i]=elev(v.v[i]);
  p.p[i].x=1+4*p.p[i].x/Pi;
  p.p[i].y=5-5*p.p[i].y;
  }
return(p);
}


/*****************************************************
This function takes a point on the Cliffird torus
and converts it to a Lorenz-null vector in S^3
*****************************************************/

vector ex(p)
point p;
{
vector v;
double cos(), sin(), sqrt();
double sq;
sq=sqrt(2.0);
v.a.x=cos(p.x)/sq;
v.a.y=sin(p.x)/sq;
v.b.x=cos(p.y)/sq;
v.b.y=sin(p.y)/sq;
v.c.x=1;
v.c.y=0;
return(v);
}


/**********************************************
dot products:  The ordinary Hermitian form
And the Lorenz-Hermitian form
********************************************/

complex dot(v,w)
vector v,w;
{
complex d;
d=plus(plus(times(v.a,w.a),times(v.b,w.b)),times(v.c,w.c));
return(d);
}


complex hdot(v,w)
vector v,w;
{
vector cw;
complex d;
cw=conj(w);
d=minus(plus(times(v.a,cw.a),times(v.b,cw.b)),times(v.c,cw.c));
return(d);
}



/**apply a matrix to a vector***/

vector apply(m,v)
matrix m;
vector v;
{
vector w;
w.a=dot(m.a,v);
w.b=dot(m.b,v);
w.c=dot(m.c,v);
return(w);
}

/********************************************
This routine takes a curve of points, and converts
it to a circle containing the first three points
of the curve.   It is only used for curves which
consist of exactly three points.
********************************************/

arc circle(x)
curve x;
{
double x1,x2,y1,y2,x3,y3;
double sqrt(), atan2();
double a1,a2,a3;
arc s;
x1=x.p[1].x;
y1=x.p[1].y;
x2=x.p[2].x;
y2=x.p[2].y;
x3=x.p[3].x;
y3=x.p[3].y;
s.p.x=(x2*x2*y1-x3*x3*y1-x1*x1*y2+x3*x3*y2 - 
 y1*y1*y2+y1*y2*y2+x1*x1*y3-x2*x2*y3 + 
 y1*y1*y3-y2*y2*y3-y1*y3*y3+y2*y3*y3)/
 (2*(x2*y1-x3*y1-x1*y2+x3*y2+x1*y3-x2*y3));
s.p.y=(x1*x1*x2-x1*x2*x2-x1*x1*x3+x2*x2*x3 + 
 x1*x3*x3-x2*x3*x3+x2*y1*y1-x3*y1*y1- 
 x1*y2*y2+x3*y2*y2+x1*y3*y3-x2*y3*y3)/
 (2*(x2*y1-x3*y1-x1*y2+x3*y2+x1*y3-x2*y3));
s.r=sqrt((s.p.x-x1)*(s.p.x-x1)+(s.p.y-y1)*(s.p.y-y1));
a1=new_atan(x1-s.p.x,y1-s.p.y);
a2=new_atan(x2-s.p.x,y2-s.p.y);
a3=new_atan(x3-s.p.x,y3-s.p.y);
if(a1<=a3)
   {
   if(a2>=a1&a2<a3)  {s.t1=a1;s.t2=a3;}
   if(a2>=a3)        {s.t1=a3;s.t2=a1+2*Pi;}
   if(a2<=a1)        {s.t1=a3;s.t2=a1+2*Pi;}
   }
if(a3<=a1)
   {
   if(a2>=a3&a2<=a1) {s.t1=a3;s.t2=a1;}
   if(a2<=a3)        {s.t1=a1;s.t2=a3+2*Pi;}
   if(a2>=a1)        {s.t1=a1;s.t2=a3+2*Pi;}
   }
if((x1-x2)*(x1-x2)+(y1-y2)*(y1-y2)<.0000001)
{s.p.x=0;s.p.y=0;s.r=0;s.t1=0;s.t2=0;}
return(s);
}

/********************************************
This program converts a circular arc to one
which can be drawn by Tcl
*********************************************/

tcl_arc arc_to_tcl(a)
arc a;
{
tcl_arc ta;
ta.p.x=5*(a.p.x-a.r)+5;
ta.p.y=5-5*(a.p.y-a.r);
ta.q.x=5*(a.p.x+a.r)+5;
ta.q.y=5-5*(a.p.y+a.r);
ta.a1=180.0*a.t1/Pi;
ta.a2=180.0*a.t2/Pi;
ta.a2=ta.a2-ta.a1;
return(ta);
}

vector pro(v)
vector v;
{
vector w;
w.a=divide(v.a,v.c);
w.b=divide(v.b,v.c);
w.c=divide(v.c,v.c);
return(w);
}

/*Matrices:   

Each matrix, say x() is followed by a
companion xx(), which applies the matrix x() to a
list of vectors (called a line).  The matrix
t() is a special instance of the more general
matric u() */


/*This is complex conjugation, followed by swapping the
coordinates*/

vector c(v,def)
vector v;
deformation def;
{
vector w;
w.a=conjugate(v.b);
w.b=conjugate(v.a);
w.c=conjugate(v.c);
return(w);
}

line cc(v,def)
line v;
deformation def;
{
int i;
line w;
w.l=v.l;
for(i=1;i<=v.l;++i)
w.v[i]=c(v.v[i],def);
return(w);
}

/*This is the element 0 in the program surface.tcl*/


vector t0(v,def)
vector v;
deformation def;
{
vector w;
matrix m0;
m0.a.a.x=0;               m0.a.a.y=0;
m0.a.b.x=1;               m0.a.b.y=0;
m0.a.c.x=0;               m0.a.c.y=0;
m0.b.a.x=1;               m0.b.a.y=0;
m0.b.b.x=0;               m0.b.b.y=0;
m0.b.c.x=0;               m0.b.c.y=0;
m0.c.a.x=0;               m0.c.a.y=0;
m0.c.b.x=0;               m0.c.b.y=0;
m0.c.c.x=1;               m0.c.c.y=0;
w=apply(m0,v);
return(w);
}


line tt0(v,def)
line v;
deformation def;
{
int i;
line w;
w.l=v.l;
for(i=1;i<=v.l;++i)
w.v[i]=t0(v.v[i],def);
return(w);
}


/*This is the element 1 in the program surface .tcl*/

vector t1(v,def)
vector v;
deformation def;
{
double t;
vector w;
matrix m1;
double cos(), sin(), sqrt();
t=def.t;

m1.a.a.x=-1;                      m1.a.a.y=0;
m1.a.b.x=0;                       m1.a.b.y=0;
m1.a.c.x=0;                       m1.a.c.y=0;
m1.b.a.x=0;                       m1.b.a.y=0;
m1.b.b.x=3;                       m1.b.b.y=0;
m1.b.c.x=-sqrt(8.0)*sin(t);       m1.b.c.y=sqrt(8.0)*cos(t);
m1.c.a.x=0;                       m1.c.a.y=0;
m1.c.b.x=sqrt(8.0)*sin(t);        m1.c.b.y=sqrt(8.0)*cos(t);
m1.c.c.x=-3;                      m1.c.c.y=0;
w=apply(m1,v);
return(w);
}

line tt1(v,def)
line v;
double def;
{
int i;
line w;
w.l=v.l;
for(i=1;i<=v.l;++i)
w.v[i]=t1(v.v[i],def);
return(w);
}


/*This is the element 2 in the program surface.tcl*/

vector t2(v,def)
vector v;
deformation def;
{
double t;
vector w;
w=t0(v,def);
w= c(w,def);
w=t1(w,def);
w= c(w,def);
w=t0(w,def);
return(w);
}

line tt2(v,def)
line v;
deformation def;
{
int i;
line w;
w.l=v.l;
for(i=1;i<=v.l;++i)
w.v[i]=t2(v.v[i],def);
return(w);
}

/*This routine computes the fixed point
of the composition h=t1 t2*/

complex fix(t)
double t;
{
deformation temp;
vector v;
int i;
temp.t=t;
v.a.x=1.0;
v.a.y=0.0;
v.b.x=1.0;
v.b.y=0.0;
v.c.x=1.0;
v.c.y=1.0;
for(i=1;i<50;++i)
   {
   v=t1(v,temp);
   v=t2(v,temp);
   v=pro(v);
   }
return(v.b);
}

/*These routines compute the fixed and piercing points s
of the composition g=t1 t2 t0*/

vector eig(t)
double t;
{
deformation temp;
vector v;
int i;
temp.t=t;
v.a.x=1.0;
v.a.y=0.0;
v.b.x=1.0;
v.b.y=0.0;
v.c.x=1.0;
v.c.y=1.0;
for(i=1;i<1000;++i)
   {
   v=t1(v,temp);
   v=t2(v,temp);
   v=t0(v,temp);
   v=pro(v);
   }
return(v);
}




complex pierce1(t)
double t;
{
vector v;
double test;
complex c1,c2,c;
double sqrt(),d1,d2,d3;
test=1.417099-t;
v=eig(t);
c1=minus(v.a,conjugate(v.b));
c2.x=v.a.x*v.a.x+v.a.y*v.a.y-v.b.x*v.b.x-v.b.y*v.b.y;
c2.y=0;
c=divide(c1,c2);
d1=c.x*c.x+c.y*c.y;
d2=(c.x+c.y*sqrt(2.0*d1-1.0))/(2.0*d1);
d3=sqrt(0.5-d2*d2);
c.x=d2;
c.y=-d3;
if(test<.0001) 
   {c.x=.698771;
    c.y=-.108253;}
return(c);
}

complex pierce2(t)
double t;
{
vector v;
complex c1,c2,c;
double sqrt(),d1,d2,d3;
double test;
v=eig(t);
c1=minus(v.a,conjugate(v.b));
c2.x=v.a.x*v.a.x+v.a.y*v.a.y-v.b.x*v.b.x-v.b.y*v.b.y;
c2.y=0;
c=divide(c1,c2);
d1=c.x*c.x+c.y*c.y;
d2=(c.x-c.y*sqrt(2.0*d1-1.0))/(2.0*d1);
d3=sqrt(0.5-d2*d2);
c.x=d2;
c.y=d3;
test=1.417099-t;
if(test<.0001) {c.x=sqrt(5.0)/4.0;c.y=sqrt(3.0)/4.0;}
return(c);
}

/*These are the functions used to define the PLAN and ELEV
coordinate systems*/

vector pre_q(v,def)
vector v;
deformation def;
{
vector w;
complex c1,c2,cc1,cc2;
double sqrt();
c1.x=sqrt(5.0)/4.0;
c1.y=sqrt(3.0)/4.0;
c2=def.p[2];
cc1=conjugate(c1);
cc2=conjugate(c2);
w.a=divide(times(v.a,c1),c2);
w.b=divide(times(v.b,cc1),cc2);
w.c=v.c;
return(w);
}

vector ipre_q(v,def)
vector v;
deformation def;
{
vector w;
complex c1,c2,cc1,cc2;
double sqrt();
c1.x=sqrt(5.0)/4.0;
c1.y=sqrt(3.0)/4.0;
c2=def.p[2];
cc1=conjugate(c1);
cc2=conjugate(c2);
w.a=divide(times(v.a,c2),c1);
w.b=divide(times(v.b,cc2),cc1);
w.c=v.c;
return(w);
}



vector zz_q(v,def)
vector v;
deformation def;
{
vector w;
matrix st;

st.a.a.x=-sqrt(5.0);
st.a.a.y=-sqrt(3.0)/5.0;
st.a.b.x=-sqrt(5.0);
st.a.b.y= sqrt(3.0)/5.0;
st.a.c.x= 16.0/5.0;
st.a.c.y= 0;

st.b.a.x= 9/sqrt(5.0);
st.b.a.y=-3*sqrt(3.0)/5;
st.b.b.x= 9/sqrt(5.0);
st.b.b.y= 3*sqrt(3.0)/5;
st.b.c.x=-27.0/5.0;
st.b.c.y= 0;

st.c.a.x=-27*sqrt(5.0)/10;
st.c.a.y= 27*sqrt(3.0)/10;
st.c.b.x=-27*sqrt(5.0)/10;
st.c.b.y=-27*sqrt(3.0)/10;
st.c.c.x= 54.0/5.0;
st.c.c.y= 0;

w=apply(st,v);
return(w);
}

vector izz_q(v,def)
vector v;
deformation def;
{
vector w;
matrix st;

st.a.a.x= sqrt(5.0)/4.0;
st.a.a.y= sqrt(3.0)/4.0;
st.a.b.x= sqrt(5.0)/3.0;
st.a.b.y= 1/(sqrt(3.0)*3.0);
st.a.c.x= 5.0*sqrt(5.0)/54;
st.a.c.y=-1/(sqrt(3.0)*18);

st.b.a.x= sqrt(5.0)/4.0;
st.b.a.y=-sqrt(3.0)/4.0;
st.b.b.x= sqrt(5.0)/3.0;
st.b.b.y=-1/(sqrt(3.0)*3.0);
st.b.c.x= 5.0*sqrt(5.0)/54;
st.b.c.y= 1/(sqrt(3.0)*18);

st.c.a.x=1;
st.c.a.y=0;
st.c.b.x=1;
st.c.b.y=0;
st.c.c.x=8.0/27.0;
st.c.c.y=0;

w=apply(st,v);
return(w);
}

vector post_q(v,def)
vector v;
deformation def;
{
matrix m;
complex c;
double r;
vector w,ww;
c=def.p[1];
w.a=c;
w.b=conjugate(c);
w.c.x=1;w.c.y=0;
w=pre_q(w,def);
w=zz_q(w);  
w=pro(w);
r=w.b.x;
m.a.a.x=1.0;
m.a.a.y=0.0;
m.a.b.x=-2.0*r;
m.a.b.y=0.0;
m.a.c.x=r*r;
m.a.c.y=0.0;
m.b.a.x=0.0;
m.b.a.y=0.0;
m.b.b.x=1.0;
m.b.b.y=0.0;
m.b.c.x=-1.0*r;
m.b.c.y=0.0;
m.c.a.x=0.0;
m.c.a.y=0.0;
m.c.b.x=0.0;
m.c.b.y=0.0;
m.c.c.x=1.0;
m.c.c.y=0.0;
ww=apply(m,v);
return(ww); 
}

vector ipost_q(v,def)
vector v;
deformation def;
{
matrix m;
complex c;
double r;
vector w,ww;
c=def.p[1];
w.a=c;
w.b=conjugate(c);
w.c.x=1;w.c.y=0;
w=pre_q(w,def);
w=zz_q(w);  
w=pro(w);
r=w.b.x;
m.a.a.x=1.0;
m.a.a.y=0.0;
m.a.b.x=2.0*r;
m.a.b.y=0.0;
m.a.c.x=r*r;
m.a.c.y=0.0;
m.b.a.x=0.0;
m.b.a.y=0.0;
m.b.b.x=1.0;
m.b.b.y=0.0;
m.b.c.x=1.0*r;
m.b.c.y=0.0;
m.c.a.x=0.0;
m.c.a.y=0.0;
m.c.b.x=0.0;
m.c.b.y=0.0;
m.c.c.x=1.0;
m.c.c.y=0.0;
ww=apply(m,v);
return(ww);
}

vector q(v,def)
vector v;
deformation def;
{
double test;
vector w;
w=pre_q(v,def); 
w=zz_q(w,def);
w=post_q(w,def);   
return(w); 
}

vector iq(v,def)
vector v;
deformation def;
{
double test;
vector w;
w=ipost_q(v,def);
w=izz_q(w,def); 
w=ipre_q(w,def); 
return(w); 
}


line qq(v,def)
line v;
deformation def;
{
int i;
line w;
w.l=v.l;
for(i=1;i<=v.l;++i)
w.v[i]=q(v.v[i],def);
return(w);
}

line iqq(v,def)
line v;
deformation def;
{
int i;
line w;
w.l=v.l;
for(i=1;i<=v.l;++i)
w.v[i]=iq(v.v[i],def);
return(w);
}


/*Making the Pieces*/


/*Finding the isometric circle*/

double isometric(t)
double t;
{
double ii,a1,a2,a3,sin(),cos(),aaa;
complex c;
deformation temp;
int i;
vector v;
temp.t=t;
a1=.1;
a2=.502344;
for(i=0;i<=100;++i)
  {
  a3=a1*.5+a2*.5;
  v.a.x=1.0;
  v.a.y=0.0;
  v.b.x=cos(a3);
  v.b.y=sin(a3);
  v.c.x=sqrt(2.0);
  v.c.y=0.0;
  v=t2(v,temp);
  v=t1(v,temp);
  v=pro(v); 
  c=divide(v.a,v.b);
  aaa=c.x*c.x+c.y*c.y;
  if(aaa>1.0) {a1=a1;a2=a3;}
  if(aaa<=1.0) {a1=a3;a2=a2;}
  }
return(a3);
}


vector interpolate(v,w,t)
vector v,w;
double t;
{
complex si,co,dd;
double sin(),cos();
vector xx;
dd=hdot(v,w);
co.x=cos(t);
co.y=0.0;
si.x=0.0;
si.y=sin(t);
si=times(dd,si);
xx.a=plus(times(co,v.a),times(si,w.a));
xx.b=plus(times(co,v.b),times(si,w.b));
xx.c=plus(times(co,v.c),times(si,w.c));
return(xx);
}

/*given two points on the Clifford torus*/
/*connect returns a null vector which interpolates*/
/* between them*/

line connect(p,q,n)
point p,q;
int n;
{
line z;
point h;
int i;
double ii,nn;
z.l=n+1;
for(i=0;i<=n;++i)
    {
    ii=i;nn=n;
    h.x=(1-ii/nn)*p.x+(ii/nn)*q.x;
    h.y=(1-ii/nn)*p.y+(ii/nn)*q.y;
    z.v[i+1]=ex(h); 
    }
return(z);
}


/* Given two null vectors, alt_connect returns another 
   null vector in the same complex line.  It has several 
   different options*/


line alt_connect(s,a,b,n)
point a,b;
int s,n;
{
vector va,vb,vc;
line z;
int i;
double ii,nn;
z.l=n+1;
va=ex(a);
vb=ex(b);
for(i=0;i<=n;++i)
    {
    ii=i; nn=n;
    if(s==0)
    {vc=pro(interpolate(va,vb,(Pi/2)*(ii/nn)*(ii/nn)));}
    if(s==1)
    {vc=pro(interpolate(va,vb,(-Pi/2)*(ii/nn)*(ii/nn)));}
    z.v[i+1]=vc;
    }
return(z);
}


vector vec_cone(v,r,def)
vector v;
double r;
deformation def;
{
vector w1,w2;
w1=pro(q(v,def));
w2.a.x=r*r*w1.a.x;
w2.a.y=    w1.a.y;
w2.b.x=  r*w1.b.x;
w2.b.y=  r*w1.b.y;
w2.c.x=1;
w2.c.y=0;
w2=iq(w2,def);
return(w2);
}


line line_cone(pos,v,density,length,def)
vector v;
double pos;
double density;
int length;
deformation def;
{
line x;
int i;
double rr,dd,ii;
for(i=0;i<=length;++i)
   {
   ii=i;
   dd=density;
   rr=pos+(ii/dd)*(ii/dd)*(ii/dd);
   x.v[i+1]=vec_cone(v,rr,def);
   }
x.l=length+1;
return(x);
 }


piece cone(pos,x,density,length,def)
double pos;
line x;
int length;
double density;
deformation def;
{
piece p;
int i;
p.l=x.l;
for(i=1;i<=x.l;++i)
  p.x[i]=line_cone(pos,x.v[i],density,length,def);
return(p);
}


/******************SURFACE PIECES******************/


/* along and trans refer to vertical and horizontal
   density--not to the actual dimension of the piece */


/********************************************
Scheme for rec_piece:
         a    b     ----0

         b    c     ----w2
**********************************************/


piece rec_piece(a,b,c,d,trans,along,def)
double a,b,c,d;
int trans, along;
deformation def;
{
point p,q;
int i;
double ii,nn,w2;
piece r;
w2=def.w[1];
for(i=0;i<=trans;++i)
  {
  ii=i;nn=trans;
  p.x=(1-ii/nn)*a+(ii/nn)*c;
  p.y=2*Pi-ii*w2/nn;
  q.x=(1-ii/nn)*b+(ii/nn)*d;
  q.y=2*Pi-ii*w2/nn;
  r.x[i+1]=connect(p,q,along); 
  }
  p.x=a;p.y=2*Pi;q.x=c;q.y=2*Pi-w2;
  r.x[trans+2]=connect(p,q,along);
  p.x=b;p.y=2*Pi;q.x=d;q.y=2*Pi-w2;
  r.x[trans+3]=connect(p,q,along);
r.l=trans+3;
return(r);
}


piece poly0(trans,along,def)
int trans, along;
deformation def;
{
double w2,w3, atan(); 
piece r;
w2=def.w[1];
w3=def.w[2];
r=rec_piece(0.0,w2,0.0,w2,trans,along,def);
return(r);
}


piece poly1(trans,along,def)
int trans, along;
deformation def;
{
double w2,w3, atan(); 
piece r;
w2=def.w[1];
w3=def.w[2];
r=rec_piece(w2,w3,w2,w3,trans,along,def);
return(r);
}

piece poly2(trans,along,def)
int trans, along;
deformation def;
{
double w2,w3, atan(); 
piece r;
w2=def.w[1];
w3=def.w[2];
r=rec_piece(w3,2*w3,w3,2*w3-w2,trans,along,def);
return(r);
}

piece poly3(trans,along,def)
int trans, along;
deformation def;
{
double w2,w3, atan(); 
piece r;
w2=def.w[1];
w3=def.w[2];
r=rec_piece(2*w3,2*Pi-2*w3,2*w3-w2,2*Pi-2*w3-w2,trans,along,def);
return(r);
}


piece poly4(trans,along,def)
int trans, along;
deformation def;
{
double w2,w3, atan(); 
piece r;
w2=def.w[1];
w3=def.w[2];
r=rec_piece(2*Pi-2*w3,2*Pi-w3,2*Pi-2*w3-w2,2*Pi-w3,trans,along,def);
return(r);
}

piece poly5(trans,along,def)
int trans,along;
deformation def;
{
double w2,w3, atan(); 
piece r;
w2=def.w[1];
w3=def.w[2];
r=rec_piece(2*Pi-w3,2*Pi-w2,2*Pi-w3,2*Pi-w2,trans,along,def);
return(r);
}

piece poly6(trans,along,def)
int trans,along;
deformation def;
{
double w2,w3, atan(); 
piece r;
w2=def.w[1];
w3=def.w[2];
r=rec_piece(2*Pi-w2,2*Pi,2*Pi-w2,2*Pi,trans,along,def);
return(r);
}



piece poly7(trans,along,def)
int trans,along;
deformation def;
{
point p,q;
int m1,m2;
line x;
piece ppp;
double w2,w5;
double aa,tt;
double atan(),sqrt();
tt=trans;
aa=along;
w2=def.w[1];
w5=2.0*Pi-w2;
p.x=w2;p.y=w5;
q.x=w5;q.y=w5;
x=alt_connect(0,p,q,trans);
ppp=cone(1.0,x,aa*.3,along,def);   /*(line,double-density,int-length)*/
ppp.x[ppp.l+1]=alt_connect(0,p,q,200);
ppp.l=ppp.l+1;
return(ppp);
}

piece poly8(trans,along,def)
int trans,along;
deformation def;
{
point p,q;
int m1,m2;
line x;
piece ppp;
double w2,w5,tt,aa;
double atan(),sqrt();
tt=trans;
aa=along;
w2=def.w[1];
w5=2.0*Pi-w2;
p.x=w5;p.y=w5;
q.x=Pi;q.y=Pi;
x=alt_connect(1,p,q,trans);
ppp=cone(1.0,x,aa*.3,along,def);   /*(line,double-density,int-length)*/
ppp.x[ppp.l+1]=alt_connect(1,p,q,200);
ppp.l=ppp.l+1;
return(ppp);
}

piece poly9(z)
complex z;
{
double cos(),sin(),sqrt(),ii;
piece ppp;
int i;
ppp.l=2;

for(i=1;i<=100;++i)
   {
   ii=i;
   ppp.x[1].v[i].a.x=cos(2*Pi*ii/100);
   ppp.x[1].v[i].a.y=0;
   ppp.x[1].v[i].b.x=sin(2*Pi*ii/100);
   ppp.x[1].v[i].b.y=0;
   ppp.x[1].v[i].c.x=1.0;
   ppp.x[1].v[i].c.y=0;
   }
ppp.x[1].l=100;

ppp.x[2].v[1].a.x=0.0;
ppp.x[2].v[1].a.y=0.0;
ppp.x[2].v[1].b.x=z.x;
ppp.x[2].v[1].b.y=z.y;
ppp.x[2].v[1].c.x=1.0;
ppp.x[2].v[1].c.y=0.0;

ppp.x[2].v[2].a.x=0.0;
ppp.x[2].v[2].a.y=0.0;
ppp.x[2].v[2].b.x=(1.0-z.y)/(z.x);
ppp.x[2].v[2].b.y=0.0;
ppp.x[2].v[2].c.x=1.0;
ppp.x[2].v[2].c.y=0.0;

ppp.x[2].v[3].a.x=0.0;
ppp.x[2].v[3].a.y=0.0;
ppp.x[2].v[3].b.x=z.x;
ppp.x[2].v[3].b.y=-z.y;
ppp.x[2].v[3].c.x=1.0;
ppp.x[2].v[3].c.y=0.0;
ppp.x[2].l=3;

return(ppp);
}


/*The Group Action*/

line sym(d,x,def)
int d;
line x;
deformation def;
{
line y;
y=x;
if(d==1) y=y;
if(d==3) y=cc(y,def);
if(d==2) y=tt0(y,def);
if(d==4) y=tt0(cc(y,def),def);
return(y);
}

line act(g,x,def)
element g;
line x;
deformation def;
{
int k;
line y;
y=x;
for(k=1;k<=g.l;++k)
   {
   if(g.w[k]==0) y=tt0(y,def);
   if(g.w[k]==1) y=tt1(y,def);
   if(g.w[k]==2) y=tt2(y,def);
   }
return(y);
}

/*Conversion to TCL Types*/

tcl_arc plan_arc(x,def)
line x;
deformation def;
{
line y;
curve c;
arc a;
tcl_arc ta;
y=qq(x,def);
c=line_to_plan(y);
a=circle(c);
ta=arc_to_tcl(a);
return(ta);
}

tcl_arc hopf_arc(x)
line x;
{
line y;
curve c;
arc a;
tcl_arc ta;
y=x;
c=line_to_hopf(y);
a=circle(c);
ta=arc_to_tcl(a);
return(ta);
}


/*Interface to TCL*/


tcl_arc_write(type,loc,joe,a,u)
FILE *joe;
tcl_arc a;
color u;
int type, loc;
{
fprintf(joe,"1\n");
fprintf(joe,"%d%d\n",type,loc);
fprintf(joe,"%c%c%c%c%c%c\n",
u.c[0],u.c[1],u.c[2],u.c[3],u.c[4],u.c[5]);
fprintf(joe,"%lf%\n%lf\n%lf\n%lf\n%lf\n%lf\n",
a.p.x,a.p.y,a.q.x,a.q.y,a.a1,a.a2);
}


tcl_curve_write(type,loc,joe,h,u)
FILE *joe;
curve h;
color u;
int type,loc;
{
int i;
for(i=1;i<h.l;++i)
  {
  fprintf(joe,"2\n");
  fprintf(joe,"%d%d\n",type,loc);
  fprintf(joe,"%c%c%c%c%c%c\n",
  u.c[0],u.c[1],u.c[2],u.c[3],u.c[4],u.c[5]);
  fprintf(joe,"%lf\n%lf\n%lf\n%lf\n",
  h.p[i].x,h.p[i].y,h.p[i+1].x,h.p[i+1].y);
  }
}


/* Main Routine */

main()
{
tcl_arc a;
int c,d,j,choice,along,trans,count,type,loc,par,math_option;
double i1,i2, parameter, atan(), sqrt();
complex axis;
piece p;
line x;
curve tc;
element g;
color u;
deformation def;
FILE *fopen(), *output, *input, *coord;
input=fopen("tcl_to_C1","r");
coord=fopen("tcl_to_C2","r");
output=fopen("C_to_tcl","w");

/******setting the parameters****/

fscanf(coord,"%d",&choice); 
fscanf(coord,"%d",&along); 
fscanf(coord,"%d",&trans); 
fscanf(coord,"%d",&par); 
i1=par/20.0;
i2=1.0-i1;
parameter=i1*atan(sqrt(35.0))+i2*atan(sqrt(125.0/3.0));
def.t=parameter;
def.p[1]=pierce1(parameter);
def.p[2]=pierce2(parameter);
/*def.w[1]=isometric(parameter) is an alternate choice*/
def.w[1]=Pi/12;
def.w[2]=new_atan(def.p[2].x,def.p[2].y);
axis=fix(parameter);

/*********************************/

while((c=getc(input)) != EOF)
  {
  u.c[0]=c;
  for(j=1;j<=5;++j)
    u.c[j]=getc(input);
  c=getc(input);
  c=c-'0';
  type=c;
  if(choice<=2)
    {
    if(type==0) {p=poly0(trans,2,def);}
    if(type==1) {p=poly1(trans,2,def);}
    if(type==2) {p=poly2(trans,2,def);}
    if(type==3) {p=poly3(trans,2,def);}
    if(type==4) {p=poly4(trans,2,def);}
    if(type==5) {p=poly5(trans,2,def);}
    if(type==6) {p=poly6(trans,2,def);}
    if(type==7) {p=poly7(trans,3*along,def);}
    if(type==8) {p=poly8(trans,3*along,def);}
    if(type==9) {p=poly9(axis);}
    } 


   if(choice==3)
    {
    if(type==0) {p=poly0(trans,along,def);}
    if(type==1) {p=poly1(trans,along,def);}
    if(type==2) {p=poly2(trans,along,def);}
    if(type==3) {p=poly3(trans,along,def);}
    if(type==4) {p=poly4(trans,along,def);}
    if(type==5) {p=poly5(trans,along,def);}
    if(type==6) {p=poly6(trans,along,def);}
    if(type==7) {p=poly7(trans,3*along,def);} 
    if(type==8) {p=poly8(trans,3*along,def);} 
    } 
  d=getc(input)-'0';
  loc=d;
  count=1;
     while((c=getc(input)) != 10)
       {
       g.w[count]=c-'0';  
       ++count;
       }
  g.l=count-1; 
  for(j=1;j<=p.l;++j)
     {
     x=p.x[j];
     x=sym(d,x,def);
     x=act(g,x,def);
     if((choice==1) && (type<7))
       {
       a=plan_arc(x,def);
       tcl_arc_write(type,loc,output,a,u);
       }

     if((choice==1) && (type>6) && (type<9))
       {
       tc=line_to_plan_tcl(qq(x,def));
       tcl_curve_write(type,loc,output,tc,u);
       }
     if((choice==2) && (type<7))
       {
       a=hopf_arc(x);
       tcl_arc_write(type,loc,output,a,u);
       }
     if((choice==2) && (type>6) & (type<9))
       {
       tc=line_to_hopf_tcl(x);
       tcl_curve_write(type,loc,output,tc,u);
       }
     if(choice==3)
       {
       tc=line_to_elev_tcl(qq(x,def));
       tcl_curve_write(type,loc,output,tc,u);
       }
     if((choice==2) && (j==1) && (type == 9))
       {  
       tc=line_to_hopf_tcl(x);
       tcl_curve_write(type,loc,output,tc,u);
       }
     if((choice==2) && (j==2) && (type == 9))
       {
       a=hopf_arc(x);
       tcl_arc_write(type,loc,output,a,u);
       }
     }
  }


fclose(output);
fclose(input);
fclose(coord);
}
