#! /usr/local/bin/wish  -f
########################################################
#                      Dented Tori
#               By Richard Evan Schwartz
#                University of Maryland
#                  res@math.umd.edu
#                      Sept 6, 1997
#                 Copyright R.E.S. 1997
#
##########################################################
#This program illustrates my recent proof of the
#Goldman-Parker conjecture on complex hyperbolic ideal
#triangle groups.   To get a fuller explanation,
#install and run the program, as directed below.
############################################################



############################################################
#         INSTALLATION/RUNNING  INSTRUCTIONS
#1. Call this file surface.tcl
#2. Usually, Tcl is stored in the directory /usr/local/bin.
#   If this is not the case, you have to alter the top
#   line of this program so that it points to where Tcl
#   is stored. 
#3. Type chmod +x surface.tcl to make the file executable.
#   It is also a good idea to type chmod -w surface2.tcl
#   to make the file "read-only".
#4. Type surface.tcl to run the program.
#5. During its run, this program compiles and runs the program
#   surface.c  You need not compile surface.c yourself
#6. If you exit this program unnaturally--i.e.
#   by some other method than the provided exit
#   button--then the following files will exist:
#   tcl_to_C1   (a data file)
#   tcl_to_C2   (a data file)
#   C_to_tcl   (a data file)
#   delete them as you see fit.
#################################################


#########################################
############BOARD SETUP#######################
#########################################
proc board_layout {} {
frame .frame0 
frame .frame1   
frame .frame2 
canvas .plan  -width 10c -height 10c 
canvas .hopf  -width 10c -height 10c
canvas .elev  -width 10c -height 10c
canvas .torus -width 10c -height 10c 
pack .frame0
pack .frame1  .frame2 -in .frame0 -side left
pack .hopf  .torus  -in .frame1 -side top
pack .plan  .elev  -in .frame2 -side top
.plan create rectangle -1c -1c 11c 11c -fill black 
.elev create rectangle -1c -1c 11c 11c  -fill black 
.hopf create rectangle -1c -1c 11c 11c -fill black
.torus create rectangle -1c -1c 11c 11c -fill black -tag back
}

proc hopf_back_init {} {
global OPTION COL COL_COUNT
.hopf delete border
if {$OPTION !=1} {
.hopf create rectangle -1c -1c 11c 11c -fill #040  -tag border
.hopf create oval 0c 0c 10c 10c -fill black -tag back
.hopf addtag im_hopf withtag back
.hopf bind border <Button-2> {
.hopf itemconfigure border -fill #$COL($COL_COUNT)}
.hopf bind back <Button-2> {
.hopf itemconfigure back -fill #$COL($COL_COUNT)}
}}

proc elev_back_init {} {
global OPTION 
.elev delete border
if {$OPTION !=1} {
.elev create rectangle 0c 0c 10c 10c -fill #004 -tag border
.elev create rectangle 1c -20c 9c 30c -fill black -tag back
.elev addtag im_elev withtag back
.elev bind border <Button-2> {
.elev itemconfigure border -fill #$COL($COL_COUNT)}
.elev bind back <Button-2> {
.elev itemconfigure back -fill #$COL($COL_COUNT)}
}}


proc plan_back_init {} {
global OPTION
if {$OPTION !=1} {
.plan create rectangle -10c -10c 20c 20c -fill #400 -tag back1
for {set i 0} {$i<10} {incr i} {
.plan create arc -10c -10c 20c 20c -fill black -outline black \
-start [expr 36*$i] -extent 19 -tag back2}
.plan addtag im_plan withtag back1
.plan addtag im_plan withtag back2
.plan bind back1 <Button-2> {
.plan itemconfigure back1 -fill #$COL($COL_COUNT) -outline #$COL($COL_COUNT)}
.plan bind back2 <Button-2> {
.plan itemconfigure back2 -fill #$COL($COL_COUNT) -outline #$COL($COL_COUNT)}
}}




###################################################
#############CONTROL BOARD INITIALIZATION##############
###################################################
proc torus_init {} {
global COLOR GG ON COLOR_COUNT COLOR_CHOICE COORD CURRENT_ELEMENT
global GG_count VERT hopf_SCALE plan_SCALE elev_SCALE
global COL
set hopf_SCALE 1
set plan_SCALE 1
set elev_SCALE 1
set VERT(plan) 0
set VERT(hopf) 0
set VERT(elev) 0
set GG_count(plan)  0
set GG_count(elev)  0
set GG_count(elev)  0
set CURRENT_ELEMENT none
set COORD 1
set COLOR_COUNT 1
set GG "e"
for {set i 0} {$i<10} {incr i} {
for {set j 0} {$j<10} {incr j} {
set ON(${i}${j}) 0
set COLOR(${i}${j}) 000000
}}
}



##################################
###########THE BASIC PIECES##############
##################################
proc quad {a b p1 p2 q1 q2 r1 r2 s1 s2 win num} {
set nnn $num 
if {$win == "torus"} {set nnn ""}
.$win create polygon \
${p1}c ${p2}c ${q1}c ${q2}c ${r1}c ${r2}c ${s1}c ${s2}c \
-fill black -tag X${a}${b}X$nnn 
.$win create line ${q1}c ${q2}c ${r1}c ${r2}c \
-fill white -tag edge -tag ${a}y$b$nnn -width .01c
.$win create line ${r1}c ${r2}c ${s1}c ${s2}c \
-fill white -tag edge -tag ${a}y$b$nnn  -width .01c
.$win create line ${s1}c ${s2}c ${p1}c ${p2}c \
-fill white -tag edge -tag ${a}y$b$nnn -width .01c
.$win addtag edge withtag ${a}y$b$nnn
.$win create line ${p1}c ${p2}c ${q1}c ${q2}c \
-fill white -tag ${a}y$b$nnn -width .01c
if {$win != "torus"} {
.$win addtag PS${win}${nnn} withtag X${a}${b}X$nnn
.$win addtag PS${win}${nnn} withtag ${a}y${b}$nnn
}
if {$win == "torus"} {
.$win bind X${a}${b}X <Button-1> {
color_increase
on}
.$win bind X${a}${b}X <Button-2> {
color_decrease
on}
.$win bind X${a}${b}X <Button-3> {off}
set sc .6
.$win scale X${a}${b}X 0 0 $sc $sc
.$win scale ${a}y${b} 0 0  $sc $sc
.$win move X${a}${b}X 2c 2.02c  
.$win move ${a}y${b}  2c 2.02c 
}}


############################################
##########MAKE 4 SYMMETRICALLY LOCATED PIECES#######
############################################
proc quads {a p1 p2 q1 q2 r1 r2 s1 s2 win num} {
set ss 10
set pp1 [expr $ss - $p1]
set pp2 [expr $ss - $p2]
set qq1 [expr $ss - $q1]
set qq2 [expr $ss - $q2]
set rr1 [expr $ss - $r1]
set rr2 [expr $ss - $r2]
set ss1 [expr $ss - $s1]
set ss2 [expr $ss - $s2]
quad $a 1 $p1 $p2 $q1 $q2 $r1 $r2 $s1 $s2 $win $num
quad $a 2 $pp2 $pp1 $qq2 $qq1 $rr2 $rr1 $ss2 $ss1 $win $num
quad $a 3 $p2 $p1 $q2 $q1 $r2 $r1 $s2 $s1 $win $num
quad $a 4 $pp1 $pp2 $qq1 $qq2 $rr1 $rr2 $ss1 $ss2 $win $num
}

##################################
###########LIST OF PIECES#################
##################################
proc torus_pieces {win num} {
quads 0     0.0 0.0 0.8 0.0 0.8 0.8 0.0 0.8 $win $num
quads 1     0.8 0.0 0.8 0.8 2.0 0.8 2.0 0.0 $win $num
quads 2     2.0 0.0 2.0 0.8 3.2 0.8 4.0 0.0 $win $num
quads 3     4.0 0.0 3.2 0.8 5.2 0.8 6.0 0.0 $win $num
quads 4     5.2 0.8 6.0 0.0 8.0 0.0 8.0 0.8 $win $num 
quads 5     8.0 0.8 8.0 0.0 9.2 0.0 9.2 0.8 $win $num
quads 6     9.2 0.8 9.2 0.0 10 0.0 10 0.8   $win $num
quads 7     0.8 0.8 2 2 2 2 9.2 0.8         $win $num
quads 8     9.2 0.8 2 2 2 2 5 5             $win $num
quad  9 1   -2 -0.1  -2.6 -.1 -2.6 -0.6 -2 -.6  $win $num
.$win raise edge
}


########################################
#####CHANGING THE COLOR COUNTER#############
########################################

proc color_increase {} {
global COL_COUNT
set  i1 [string range $COL_COUNT 0 0]
set  i2 [string range $COL_COUNT 2 [expr [string length $COL_COUNT]-1]]
set j2 [expr $i2]
incr j2
set temp ${i1}X$j2
if {$COL_COUNT == "2X31"} {set temp 1X8}
if {$COL_COUNT == "1X31"} {set temp 2X8}
set COL_COUNT $temp
color $temp
}

proc color_decrease {} {
global COL_COUNT
set  i1 [string range $COL_COUNT 0 0]
set  i2 [string range $COL_COUNT 2 [expr [string length $COL_COUNT]-1]]
set j2 [expr $i2]
incr j2 -1
set temp ${i1}X$j2
if {$COL_COUNT == "1X8"} {set temp 2X31}
if {$COL_COUNT == "2X8"} {set temp 1X31}
set COL_COUNT $temp
color $temp
}

############################################



#######################################
########PIECE SLECTION####################
#########################################
proc on {} {
global MANY OPTION COL_COUNT COL 
set z [.torus gettags current]
set z [string trim $z current]
set i [string range $z 1 1]
set j [string range $z 2 2]
set piece [string range $z 1 2]
if {$OPTION == 1} {on_text $piece $COL($COL_COUNT)}
if {$MANY == 0} {proof_on $i $j $COL_COUNT}

if {($MANY ==1)  & ($i<7) &  ($j ==1)} {
for {set k 1} {$k<6} {incr k} {
proof_on $k $j $COL_COUNT}
proof_on  0 3 $COL_COUNT
proof_on  6 2 $COL_COUNT}
if {($MANY ==1)  & ($i<7) &  ($j ==2)} {
for {set k 1} {$k<6} {incr k} {
proof_on $k $j $COL_COUNT}
proof_on  0 4 $COL_COUNT
proof_on  6 2 $COL_COUNT}
if {($MANY ==1)  & ($i<7) &  ($j ==3)} {
for {set k 1} {$k<6} {incr k} {
proof_on $k $j $COL_COUNT}
proof_on  0 3 $COL_COUNT
proof_on  6 4 $COL_COUNT}
if {($MANY ==1)  & ($i<7) &  ($j ==4)} {
for {set k 1} {$k<6} {incr k} {
proof_on $k $j $COL_COUNT}
proof_on  0 4 $COL_COUNT
proof_on  6 4 $COL_COUNT}
if {($MANY ==1)  & ($i>6) & ($i<9) & ($j ==1)} {
proof_on 7 1 $COL_COUNT
proof_on 7 3 $COL_COUNT
proof_on 8 1 $COL_COUNT
proof_on 8 3 $COL_COUNT
}
if {($MANY ==1)  & ($i>6) & ($i<9) & ($j ==3)} {
proof_on 7 1 $COL_COUNT
proof_on 7 3 $COL_COUNT
proof_on 8 1 $COL_COUNT
proof_on 8 3 $COL_COUNT
}
if {($MANY ==1)  & ($i>6) & ($i<9) & ($j ==2)} {
proof_on 7 2 $COL_COUNT
proof_on 7 4 $COL_COUNT
proof_on 8 2 $COL_COUNT
proof_on 8 4 $COL_COUNT
}
if {($MANY ==1)  & ($i>6) & ($i<9) & ($j ==4)} {
proof_on 7 2 $COL_COUNT
proof_on 7 4 $COL_COUNT
proof_on 8 2 $COL_COUNT
proof_on 8 4 $COL_COUNT
}

}

proc off {} {
global MANY OPTION COL_COUNT COL 
set z [.torus gettags current]
set z [string trim $z current]
set i [string range $z 1 1]
set j [string range $z 2 2]
if {$MANY == 0} {proof_off $i $j} 

if {($MANY ==1)  & ($i<7) &  ($j ==1)} {
for {set k 1} {$k<6} {incr k} {
proof_off $k $j }
proof_off  0 3 
proof_off  6 2 }
if {($MANY ==1)  & ($i<7) &  ($j ==2)} {
for {set k 1} {$k<6} {incr k} {
proof_off $k $j }
proof_off  0 4 
proof_off  6 2 }
if {($MANY ==1)  & ($i<7) &  ($j ==3)} {
for {set k 1} {$k<6} {incr k} {
proof_off $k $j }
proof_off  0 3 
proof_off  6 4 }
if {($MANY ==1)  & ($i<7) &  ($j ==4)} {
for {set k 1} {$k<6} {incr k} {
proof_off $k $j }
proof_off  0 4 
proof_off  6 4 }
if {($MANY ==1)  & ($i>6) & ($i<9) & ($j ==1)} {
proof_off 7 1 
proof_off 7 3 
proof_off 8 1 
proof_off 8 3 
}
if {($MANY ==1)  & ($i>6) & ($i<9) & ($j ==3)} {
proof_off 7 1 
proof_off 7 3 
proof_off 8 1 
proof_off 8 3 
}
if {($MANY ==1)  & ($i>6) & ($i<9) & ($j ==2)} {
proof_off 7 2 
proof_off 7 4 
proof_off 8 2 
proof_off 8 4 
}
if {($MANY ==1)  & ($i>6) & ($i<9) & ($j ==4)} {
proof_off 7 2 
proof_off 7 4 
proof_off 8 2 
proof_off 8 4 
}

}

proc proof_on {i j v} {
global COLOR ON CURRENT_ELEMENT COORD COL COL_COUNT
set_current_element
set col $COL($v)
.torus itemconfigure X${i}${j}X -fill #$col
set    ON(${i}${j}) 1
set COLOR(${i}${j}) $col
if {$COORD <= 1} {set s plan}
if {$COORD == 2} {set s hopf}
if {$COORD == 3} {set s elev}
set tag "X${CURRENT_ELEMENT}X${i}${j}X"
.$s raise $tag
.torus dtag off$tag
set memory [.torus gettags $tag] 
set match  [lsearch $memory ${tag}C*]
if {$match != -1} {
.torus dtag [lindex $memory $match]
.torus addtag ${tag}C${col} withtag back
catch {.$s itemconfigure ${tag} -fill #$col -outline #$col}}
catch {.$s itemconfigure ${tag} -fill #$col}
}

proc proof_off {i j} {
global COLOR ON CURRENT_ELEMENT COORD PSBACK
set_current_element
.torus itemconfigure X${i}${j}X -fill $PSBACK
set ON(${i}${j}) 0
set COLOR(${i}${j}) 000000
set tag "X${CURRENT_ELEMENT}X${i}${j}X"
if {$COORD <= 1} {set s plan}
if {$COORD == 2} {set s hopf}
if {$COORD == 3} {set s elev}
.$s lower $tag
set memory [.torus gettags $tag] 
set match  [lsearch $memory ${tag}C*]
if {$match != -1} {
.torus addtag off$tag withtag back}
}


##################################
########GROUP COMPOSITION#########
################################
proc compose {type} {
global GG COORD OPTION TY
if {$GG == ""} {set GG "e"}
set l [string length $GG]
set last [string range $GG 0 0]
if {($l == 1) & ($GG == $type)} {set HH "e"}
if {($l == 1) & ($GG != $type)} {
if {$GG != "e"} {set HH "$type$GG"}}
if {($l == 1) & ($GG == "e")} {set HH $type}
if {($l >  1) & ($last == $type)} {
set HH [string range $GG 1 [expr $l-1]]}
if {($l >  1) & ($last != $type)} {set HH "$type$GG"}
set GG $HH
.torus itemconfigure element -text $GG -fill yellow
set_current_element
set TY $type
if {$OPTION == 1} {
global TY
compose_text $TY}
}		   

proc decompose {s} {
global GG OPTION
set l [string length $GG]
if {$l == 1} {set GGG "e"}
if {$l >  1} {set GGG [string range $GG 1 [expr $l - 1]]}
set GG $GGG
.torus itemconfigure element -text $GG -fill yellow
set_current_element
if {$OPTION == 1} {decompose_text}
}

proc reverse {s} {
global G
set l [string length $s]
set G ""
for {set i [expr $l-1]} {$i > -1} {incr i -1} {
set ss [string range $s $i $i]
set G "$G$ss"}}



#######################################################
###CLEARING##########################
#######################################################
proc clear {s} {
global PSLABEL SCALE_plan SCALE_hopf SCALE_elev CURRENT_ELEMENT GG XGG
global XGG GG_count OPTION
set PSLABEL "yes" 
ps_label
set GG_count($s) 0
set GG "e"
set SCALE_${s} 1
set CURRENT_ELEMENT 1$s
clear_torus
.torus itemconfigure element -text "e"
for {set i 1} {$i<18} {incr i} {
set XGG(${i}${s}) ""
.torus itemconfigure ${i}${s} -text ""}
down $s
set mem [.torus gettags back]
while {[expr [lsearch $mem *$s*]] != -1} {
.torus dtag [lindex $mem [expr [lsearch $mem *$s*]]]
set mem [.torus gettags back]}
.${s} delete im_$s
.${s} create oval 0c 0c 0c 0c -fill black -tag origin
.${s} addtag im_$s withtag origin
if {$OPTION == 1} {clear_text}
${s}_back_init
}

proc clear_torus {} {
global ON COLOR PSBACK PSLINES OPTION
for {set i 0} {$i<10} {incr i} {
for {set j 0} {$j<10} {incr j} {
.torus itemconfigure X${i}${j}X -fill $PSBACK
.torus itemconfigure ${i}y${j} -fill $PSLINES
set ON($i$j) 0
set COLOR($i$j) 000000}}
}

proc partial_clear_torus {} {
global ON COLOR PSBACK PSLINES
for {set i 0} {$i<10} {incr i} {
for {set j 0} {$j<10} {incr j} {
.torus itemconfigure ${i}y${j} -fill #888
if {$ON(${i}${j}) == 0} {.torus itemconfigure X${i}${j}X -fill $PSBACK}
.torus itemconfigure ${i}y${j} -fill $PSLINES
}}
}

#################################
#####SWITCH CONTROLS##############
#################################
proc switch {s} {
global OPTION COORD START CURRENT_ELEMENT
if {$s == "plan"} {set COORD 1}
if {$s == "hopf"} {set COORD 2}
if {$s == "elev"} {set COORD 3}
down $s
if {$OPTION == 1} {switch_text}
foreach name {hopf plan elev} {
.torus raise YswitchD$name
.torus raise ${name}_cover}
.torus lower YswitchD$s 
.torus lower ${s}_cover
set_current_element
}



#####################################################
############SET CURRENT ELEMENT******************
#####################################################
proc set_current_element {} {
global GG XGG  CURRENT_ELEMENT COORD GG_count NEW_GG
set NEW_GG 1
if {$COORD == 1} {set s plan}
if {$COORD == 2} {set s hopf}
if {$COORD == 3} {set s elev}
set last [expr $GG_count($s) + 1]
set CURRENT_ELEMENT $last$s
for {set i 1} {$i<$last} {incr i} {
set FF $XGG(${i}$s)
set match [lsearch $FF $GG]
if {$match == 0} {
set NEW_GG 0
set CURRENT_ELEMENT "$i$s"}}
}

############################################
############ADD NEW ELEMENT TO LIST####################
#####################################################
proc list_update {s} {
global CURRENT_ELEMENT GG NEW_GG XGG GG_count
set_current_element
set last [expr $GG_count($s) + 1]
set XGG(${last}$s) $GG
if {$NEW_GG == 1} {
.torus itemconfigure ${last}${s} -text $GG 
incr GG_count($s)}
}


############################################
###############MEMORY################
############################################
proc memory {null} {
global COORD XGG GG CURRENT_ELEMENT NEW_GG OPTION
set vv [.torus gettags current]
set yy [lindex $vv 0]
set zz [lindex $vv 1]
set s [string trim $zz t]
set CURRENT_ELEMENT $yy
set NEW_GG 0
if {$XGG($yy) != ""} {
set GG $XGG($CURRENT_ELEMENT)
.torus itemconfigure element -text $GG -fill yellow
show_torus $s
if {$OPTION == 1} {memory_text}}
}

proc show_torus {s} {
global GG CURRENT_ELEMENT ON COLOR
clear_torus 
switch $s
set_current_element
set memory [.torus gettags back]
for {set i 0} {$i<10} {incr i} {
for {set j 0} {$j<10} {incr j} {
set tag X${CURRENT_ELEMENT}X${i}${j}X
set match1 [lsearch $memory $tag]
set match2 [lsearch $memory off$tag]
if {($match1 != -1) && ($match2 == -1)} {
set color [lindex $memory [expr [lsearch $memory ${tag}C*]]] 
set len [string length $color]
set color [string range $color [expr $len - 6] [expr $len -1]]
.torus itemconfigure X${i}${j}X -fill #$color
set ON(${i}${j}) 1
set COLOR(${i}${j}) $color
.$s itemconfigure $tag -fill #$color
}}}}


#########################################
#######TCL ----> C########################
#########################################

proc blocker {} {
global COL
for {set i 0} {$i<21} {incr i} {
set ii [expr $i + 8]
.torus create rectangle [expr .25*$i]c \
                        [expr .25*$i]c \
                        [expr 10-.25*$i]c \
                        [expr 10-.25*$i]c \
-fill #$COL(1X${ii}) -tag block
.torus bind block <Any-Motion> {.torus delete block}
}} 


proc update {s} {
global PARAMETER OPTION G GG ON COLOR WIDTH HEIGHT COORD CURRENT_ELEMENT
.plan delete pix
list_update $s
set tem [open tcl_to_C2 w]
puts $tem $COORD
puts $tem $WIDTH
puts $tem $HEIGHT
puts $tem $PARAMETER
close $tem
set file [open tcl_to_C1 w]
if {$GG == "e"} {set GG ""}
reverse $GG
set memory [.torus gettags back]
for {set i 0} {$i<10} {incr i} {
for {set j 0} {$j<10} {incr j} {
set col $COLOR(${i}${j})
set match [lsearch $memory X${CURRENT_ELEMENT}X${i}${j}X]
if {($match == -1) && ($ON(${i}${j}) == 1)} {
puts $file "${col}${i}${j}$G"
set tag X${CURRENT_ELEMENT}X${i}${j}X 
.torus addtag $tag withtag back
.torus addtag ${tag}C$col withtag back
}}}
close $file
if {$OPTION > 1} {
catch {exec surface.o}
file_read $s}
if {$GG == ""} {set GG "e"}
if {$OPTION == 1} {update_text}
}

##################################
#########C ------>TCL##############
##################################
proc file_read {s} {
global infile
set count 0
set infile [open C_to_tcl r]
while {[gets $infile line] >= 0} {
set test $line
if {$test == 1}   {arc_write $s $count}
if {$test == 2} {curve_write $s $count}
incr count
}
close $infile
}

#ARCS
proc arc_write {s count} {
global infile GG CURRENT_ELEMENT
global SCALE_plan SCALE_hopf SCALE_elev
if {$s == "plan"} {set sc $SCALE_plan}
if {$s == "hopf"} {set sc $SCALE_hopf}
if {$s == "elev"} {set sc $SCALE_elev}

set loc [.$s coords origin]
set transX [expr [lindex $loc 0]]
set transY [expr [lindex $loc 1]]
foreach name {piece s1 p1 p2 q1 q2 a1 a2} {
gets $infile line
set $name $line }
set tag X${CURRENT_ELEMENT}X${piece}X
set point [expr ($p1-$p2) * ($p1-$p2)]
if {$point>.0001}  \
{.$s create arc ${p1}c ${p2}c ${q1}c ${q2}c \
-fill #$s1  -outline #$s1 -start $a1  -extent $a2 -style arc \
-tag ${tag}Z$count}
.$s addtag $tag withtag ${tag}Z$count
.$s addtag im_$s withtag $tag
.$s scale ${tag}Z$count 0 0 $sc $sc
.$s move  ${tag}Z$count $transX $transY
}

#CURVES
proc curve_write {s count} {
global infile GG CURRENT_ELEMENT
global SCALE_plan SCALE_hopf SCALE_elev
if {$s == "plan"} {set sc $SCALE_plan}
if {$s == "hopf"} {set sc $SCALE_hopf}
if {$s == "elev"} {set sc $SCALE_elev}
set loc [.$s coords origin]
set transX [expr [lindex $loc 0]]
set transY [expr [lindex $loc 1]]
foreach name {piece s1 px py qx qy} {
gets $infile line
set $name $line}
set tag X${CURRENT_ELEMENT}X${piece}X
if {[expr ($px - $qx)*($px - $qx)<1]} {
.$s create line ${px}c ${py}c ${qx}c ${qy}c \
-fill #$s1 -tag ${tag}Z$count}
.$s addtag $tag withtag ${tag}Z$count
.$s addtag im_$s withtag $tag
.$s scale ${tag}Z$count 0 0 $sc $sc
.$s move  ${tag}Z$count $transX $transY
}


################################
######SCALING######################
################################
proc scale_init {} {
global SCALE_hopf SCALE_plan SCALE_elev
set ACOORD 0
set BCOORD 0
foreach name {plan hopf elev} {
set SCALE_$name 1
set SCALE_${name}_X 1
set SCALE_${name}_Y 1}
bind .plan <Button-1> {location %x %y
		       plan_up_scale}
bind .plan <Button-3> {location %x %y
   	      plan_dn_scale}
bind .hopf <Button-1> {location %x %y
     hopf_up_scale}
bind .hopf <Button-3> {location %x %y
	    hopf_dn_scale}
bind .elev <Button-1> {location %x %y
	    elev_up_scale}
bind .elev <Button-3> {location %x %y
            elev_dn_scale}
}

#######
proc location {a b} {
global ACOORD BCOORD
set ACOORD $a
set BCOORD $b
}
#######
proc plan_up_scale {} {
global ACOORD BCOORD SCALE SCALE_plan
set sc [expr 11.0/10.0]
if {$SCALE_plan<64} \
{set SCALE_plan [expr $SCALE_plan*$sc]
.plan scale im_plan $ACOORD $BCOORD $sc $sc}
}
#######
proc plan_dn_scale {} {
global ACOORD BCOORD SCALE SCALE_plan
set sc [expr 10.0/11.0]
if {$SCALE_plan>.5} \
{set SCALE_plan [expr $SCALE_plan*$sc]
.plan scale im_plan $ACOORD $BCOORD $sc $sc}
}
#######
proc hopf_up_scale {} {
global ACOORD BCOORD SCALE SCALE_hopf
set sc [expr 11.0/10.0]
if {$SCALE_hopf < 64} \
{set SCALE_hopf [expr $SCALE_hopf*$sc]
.hopf scale im_hopf $ACOORD $BCOORD $sc $sc}
}
######
proc hopf_dn_scale {} {
global ACOORD BCOORD SCALE SCALE_hopf
set sc [expr 10.0/11.0]
if {$SCALE_hopf>.25} \
{set SCALE_hopf [expr $SCALE_hopf*$sc]
.hopf scale im_hopf $ACOORD $BCOORD $sc $sc}
}
######
proc elev_up_scale {} {
global ACOORD BCOORD SCALE SCALE_elev
set sc [expr 11.0/10.0]
if {$SCALE_elev < 128} \
{set SCALE_elev [expr $SCALE_elev*$sc]
.elev scale im_elev $ACOORD $BCOORD $sc $sc}
}
######
proc elev_dn_scale {} {
global ACOORD BCOORD SCALE SCALE_elev
set sc [expr 10.0/11.0]
if {$SCALE_elev>.25} \
{set SCALE_elev [expr $SCALE_elev*$sc]
.elev scale im_elev $ACOORD $BCOORD $sc $sc}
}

########################
#####TEXT SCROLLING#####
########################
proc down {s} {
global VERT OPTION
set move $VERT($s)
.torus move t$s 0c ${move}c
set VERT($s) 0
if {$OPTION == 1} {down_text}
}

proc up {s} {
global VERT OPTION
if {$VERT($s) < 6} {
set VERT($s) [expr $VERT($s) + 0.2]
.torus move t$s 0c -.2c
if {$OPTION == 1} {up_text}
}}

################################
#####RENDERING OPTIONS###########
################################
proc width {n} {
global WIDTH OPTION
set WIDTH [expr round(2*pow(2,$n))]
for {set i 1} {$i <= 5} {incr i} {
.torus lower ZwidthD$i}
.torus raise ZwidthD$n 
if {$OPTION == 1} {render_text}
}

proc height {n} {
global HEIGHT OPTION
set HEIGHT [expr round(2*pow(2,$n))]
for {set i 1} {$i <= 5} {incr i} {
.torus lower ZheightD$i}
.torus raise ZheightD$n 
if {$OPTION == 1} {render_text}
}


################################
#####CLUSTERING OPTIONS###########
################################
proc many {n} {
global MANY OPTION
set MANY $n
for {set i 0} {$i <= 1} {incr i} {
.torus lower ZmanyD$i}
.torus raise ZmanyD$n 
if {$OPTION == 1} {many_text $n}
}


############################################
#################CONTROL BUTTON###############
############################################
proc control_box {x1 y1 x2 y2 col1 col2 name opt text} {
global FONT2 
.torus create rectangle ${x1}c ${y1}c ${x2}c ${y2}c \
-fill $col2 -tag Z${name}D$opt 
.torus create text [expr 0.5*$x1+0.5*$x2]c [expr 0.5*$y1 + 0.5*$y2]c \
-font $FONT2 \
-fill white -text $text -tag Z${name}D$opt
.torus create rectangle ${x1}c ${y1}c ${x2}c ${y2}c \
-fill $col1 -tag X${name}D$opt 
.torus create text [expr 0.5*$x1+0.5*$x2]c [expr 0.5*$y1 + 0.5*$y2]c \
-font $FONT2 \
-fill white -text $text -tag Y${name}D$opt
.torus addtag Y${name}D$opt withtag X${name}D$opt
.torus addtag Y${name}D$opt withtag Z${name}D$opt
.torus addtag A$col1  withtag Y${name}D$opt
.torus addtag B$col2  withtag Y${name}D$opt
.torus bind Y${name}D$opt <Any-Enter> {control_enter}
.torus bind Y${name}D$opt <Any-Leave> {control_leave}
.torus bind Y${name}D$opt <Any-ButtonPress> {control_push}
.torus bind Y${name}D$opt <Any-ButtonRelease> {control_release}
}

proc control_enter {} {
set x [.torus gettags current]
set name [lindex $x [expr [lsearch $x Y*]]]
set name [string trim $name X]
set name [string trim $name Y]
set col  [lindex $x [expr [lsearch $x B*]]]
set col [string trim $col B]
.torus itemconfigure X$name -fill $col
}
proc control_leave {} {
set x [.torus gettags current]
set name [lindex $x [expr [lsearch $x Y*]]]
set name [string trim $name X]
set name [string trim $name Y]
set col  [lindex $x [expr [lsearch $x A*]]]
set col [string trim $col A]
.torus itemconfigure X$name -fill $col
}

proc control_push {} {
set x [.torus gettags current]
set name [lindex $x [expr [lsearch $x Y*]]]
set name [string trim $name X]
set name [string trim $name Y]
set name1 [string range $name 0 [expr [string first D $name] -1]]
set name2 [string range $name [expr [string first D $name] +1] \
		              [expr [string length $name] -1]]	
.torus raise pix
if {$name1=="update"} {blocker} 
}

proc control_release {} {
set x [.torus gettags current]
set name [lindex $x [expr [lsearch $x Y*]]]
set name [string trim $name X]
set name [string trim $name Y]
set name1 [string range $name 0 [expr [string first D $name] -1]]
set name2 [string range $name [expr [string first D $name] +1] \
		              [expr [string length $name] -1]]	
$name1 $name2
}


proc color {v} {
global COL_COUNT OPTION
set COL_COUNT $v
for {set i 1} {$i <= 32} {incr i} {
.torus lower ZcolorD1X$i
.torus lower ZcolorD2X$i}
.torus raise ZcolorD$v 
if {$OPTION == 1} {color_text}
}









###########################################
############CONTROL BOXES################
###########################################
proc controls {} {
global COL
global GG XGG FONT FONT1 FONT2 FONT3
#BLUE WINDOWS
.torus create rectangle 8c .7c 9.6c 2c -fill blue 
.torus create rectangle .4c .7c 2c 2c -fill blue 
.torus create rectangle 8 8.7c 9.6c 10c -fill blue 
#MEMORY LIST
set GG "e"
foreach s {hopf plan elev} { 
for {set j 1} {$j<18} {incr j} {
set XGG(${j}${s}) "e"
.torus create text 1.5c [expr 0.6+ 0.4*$j]c \
-text "" -fill yellow -tag ${j}$s -anchor e
.torus addtag t$s withtag ${j}$s
.torus bind ${j}${s} <Any-Enter> {
.torus itemconfigure current -fill red}
.torus bind ${j}${s} <Any-Leave> {
.torus itemconfigure current -fill yellow}
.torus bind ${j}$s <Any-Button> {memory null}
}}
#CORNER CONTROL MODULE
control_box 8 0 9 .7 #f60 #0af clear plan clear 
control_box 9 0 10 .7 #f60 #0af update plan  plot
control_box 9.6 .7 10 1.35 #f60 #0af up plan "" 
control_box 9.6 1.35 10 2 #f60 #0af down plan "" 
.torus create rectangle 9.6c .7c 10c 2c -fill #700 -tag plan_cover
control_box 0 0 1 .7 #f60 #0af clear hopf clear 
control_box 1 0 2 .7 #f60 #0af update hopf plot
control_box 0 .7 .4 1.35 #f60 #0af up hopf "" 
control_box 0 1.35 .4 2 #f60 #0af down hopf "" 
.torus create rectangle 0c .7c .4c 1.97c -fill #700 -tag hopf_cover
control_box 8 8.1 9 8.7 #f60 #0af clear elev clear 
control_box 9 8.1 10 8.7 #f60 #0af update elev plot
control_box 9.6 8.7 10 9.3 #f60 #0af up elev "" 
control_box 9.6 9.3 10 10 #f60 #0af down elev "" 
.torus create rectangle 9.6c 8.7c 10c 10c -fill #700 -tag elev_cover
control_box 8 0 10 .7 #700 #f60 switch plan "PLAN"
control_box 8.05 8.1 10 8.7 #700 #f60 switch elev "ELEV"
control_box 0 0 2 .7 #700 #f60 switch hopf "HYP"
#GLOBAL OPTIONS
control_box 8.7 2.00 10 2.58 #606 #599 option tutorial tutorial 
control_box 8.7 2.58 10 3.16 #606 #599 option play  play 
control_box 8.7 3.16 10 5.7 blue red option exit  exit 
#PARAMETER OPTIONS
for {set i 0} {$i<21} {incr i} {
control_box 8.04 [expr 2 + 0.29*$i]  8.7 [expr 2.29 + 0.29*$i]  \
#909 yellow parameter $i ""
}
#PLOT DENSITY
for {set i 1} {$i<6} {incr i} {
control_box [expr $i*0.4+1.6] 1.35 [expr $i*0.4 +2] 1.97 \
#909 #ff0 width $i "" 
control_box [expr $i*0.4+1.6] .7 [expr $i*0.4+2] 1.35 \
#909 #ff0 height $i ""}
#GROUP COMPOSITION 
control_box 6.35 .7 6.9 1.35 #909 #d0d compose 0 "0" 
control_box 6.9 .7 7.45 1.35 #909 #d0d compose 1 "1" 
control_box 7.45 .7 8.0 1.35 #909 #d0d compose 2 "2" 
control_box 6.35 1.35 8.0 2 #909 #d0d decompose null "-" 
#ELEMENT DISPLAY
.torus create rectangle 2c 0c 8c .7c -fill blue 
.torus create text 7.8c .38c -anchor e -fill yellow \
-text $GG -tag element
#POSTSCRIPT
control_box 8.7 6.94 9.35 7.5  purple cyan post hopf ""
control_box 9.35 6.94 10 7.5 purple cyan post plan ""
control_box 8.7 7.5 9.35 8.1  purple cyan post torus ""
control_box 9.35 7.5 10 8.1  purple cyan post elev ""
.torus create rectangle 8.7c 5.65c 10c 6.94c \
-fill purple 
.torus create rectangle 8.88c 5.9c 9.82c 6.80c \
-fill #606 -outline white -tag pssize
.torus create rectangle 8.72c 5.67c 9.15c 6.10c \
-fill #606 -outline white -tag pslabel
.torus bind pssize <Button-2> {ps_back 0}
.torus bind pssize <Double-Button-2> {ps_back 1}
.torus bind pssize <Button-1> {ps_size 1}
.torus bind pssize <Button-3> {ps_size 0}
.torus bind pslabel <Button-1> {ps_label}
.torus bind pslabel <Button-2> {ps_labelsize 0}
.torus bind pslabel <Button-3> {ps_labelsize 1}
#CLUSTER MODE
control_box 4 .7 5.18 1 #909 #ff0 many 0 ""
control_box 5.18 .7 6.34 1 #909 #ff0 many 1 ""
#COLOR SELECTION

set COL(1X8) 000044
set COL(1X9) 000088
set COL(1X10) 0000ff
set COL(1X11) 0055ff
set COL(1X12) 0088ff
set COL(1X13) 00aaff
set COL(1X14) 00ccff
set COL(1X15) 00eeff
set COL(1X16) 00ffdd
set COL(1X17) 00ffaa
set COL(1X18) 00ff44
set COL(1X19) 00ff11
set COL(1X20) 00cc00
set COL(1X21) 008800
set COL(1X22) 004400
set COL(1X23) ccff00
set COL(1X24) ffdd00
set COL(1X25) ffbb00
set COL(1X26) ff9900
set COL(1X27) ff5500
set COL(1X28) ff0000
set COL(1X29) 990000
set COL(1X30) 770000
set COL(1X31) 440000
set COL(1X32) 110000

set COL(2X8) ffbbff
set COL(2X9) ff55ff
set COL(2X10) ff00ff
set COL(2X11) bb00bb
set COL(2X12) 880088
set COL(2X13) 660066
set COL(2X14) 440044
set COL(2X15) 5500bb
set COL(2X16) 8800ff
set COL(2X17) ffffff
set COL(2X18) eeeeee
set COL(2X19) dddddd
set COL(2X20) cccccc
set COL(2X21) bbbbbb
set COL(2X22) aaaaaa
set COL(2X23) 999999
set COL(2X24) 888888
set COL(2X25) 777777
set COL(2X26) 555555
set COL(2X27) 444444
set COL(2X28) 333333
set COL(2X29) 222222
set COL(2X30) 111111
set COL(2X31) 000000
set COL(2X32) 000000

for {set i 8} {$i<32} {incr i} {
control_box 1.3 [expr .251*$i] 1.95 [expr .251+.251*$i] \
#80e yellow color 1X$i "" 
.torus create rectangle 0c [expr .251*$i]c 1.3c [expr .251+.251*$i]c \
-fill #$COL(1X${i})} 

for {set i 8} {$i<32} {incr i} {
control_box [expr .251*$i] 8.1 [expr .251+.251*$i] 8.7 \
#80e yellow color 2X$i "" 
.torus create rectangle [expr .251*$i]c 8.7c  [expr .251+.251*$i]c 10c \
-fill #$COL(2X${i})} 



#TITLE
.torus create rectangle 4c 1c 6.35c 2c -fill blue -tag change
.torus create rectangle 0c 8c 2c 10c -fill blue
.torus create  line 8.85c 5.7c 8.7c 5.3c -fill yellow -tag logo
.torus create  line 8.55c 5.7c 8.7c 5.3c -fill yellow -tag logo
.torus create  line 8.7c 5.3c  8.3c 5c -fill yellow -tag logo
.torus create oval 8.04c 4.72c 8.34c 5.05c -fill #555 -outline yellow -tag logo
.torus create oval 8.2c 4.77c 8.3c 4.86c -fill cyan -outline black -tag logo
.torus create  line 8.4c 5.05c 8.57c 4.6c -fill yellow -tag logo
.torus create  line 8.4c 5.05c 9c 5c -fill yellow -tag logo
.torus create oval 8.6c 4.4c 9.2c 4.95c -fill red -outline cyan -tag logo
.torus create oval 8.73c 4.52c 9.07c 4.83c -fill blue -outline cyan -tag logo
.torus create text 5.17c 1.15c -anchor n -fill white \
-font $FONT2 \
-text "Dented Tori"
.torus create text 5.17c 1.92c -anchor s -fill white \
-font $FONT2 \
-text "Rich Schwartz"
#ADJUSTMENTS
.torus move thopf .35c 0c
.torus move tplan 7.95c 0c
.torus move telev 7.95c 8c
.torus move logo -7.98c 4.2c
.torus scale logo 0c 10c 1.61 1.39
}


##################################################
#################GLOBAL OPTIONS########
####################################################
proc parameter {n} { 
global OPTION PARAMETER 
set PARAMETER $n
for {set i 0} {$i <= 20} {incr i} {
.torus lower ZparameterD$i}
.torus raise ZparameterD$n 
if {$OPTION == 1} {parameter_text $PARAMETER}
}


proc option {s} {
global OPTION PSLABEL PSBACK PSLINES
set PSLABEL "yes"
ps_label 
set PSBACK black
set PSLINES white
clear_torus
if {$s == "tutorial"} {set OPTION 1}
if {$s == "play"} {set OPTION 2}
if {$s == "proof"} {set OPTION 3}
if {$s == "exit"} {clean_exit}
height 1
width 1
foreach name {hopf plan elev} {
clear $name
.torus raise YswitchD$name
.torus raise ${name}_cover}
.torus delete proof
.torus delete xtra
.elev delete control
.plan delete pix
.hopf delete text
.torus lower ZoptionDtutorial
.torus lower ZoptionDplay
.torus lower ZoptionDproof
.torus raise ZoptionD$s
catch {destroy .proof}
hopf_back_init
plan_back_init
elev_back_init

if {$s == "tutorial"} {text1}
if {$s == "play"} {
.torus raise pix}
}

proc clean_exit {} {
catch {exec rm C_to_tcl}
catch {exec rm tcl_to_C1}
catch {exec rm tcl_to_C2}
catch {exec rm surface.o}
catch {exec rm image.m}
catch {exec rm temp.ps}
exit
}

source text.tcl


########################################
###########POSTSCRIPT INCLUSION#########
########################################
proc post {s} {
global OPTION FONT1 GG_count PSBACK PSSIZE PSLABEL
if {$OPTION ==1} {post_text $s}
if {$OPTION !=1} {
if {$s !="torus"} {
.$s create text 5c 8.65c  -fill black \
-font $FONT1 -text $s -tag  ps
.$s lower ps
for {set k 1} {$k  <= $GG_count($s)} {incr k} {ps_memory $s $k}}
.$s postscript -pageheight [expr $PSSIZE*7.5]i -colormode color \
-height 10c -width 10c \
-file "temp.ps"
.$s delete ps
}}

proc ps_memory_setup {} {
global PSLABELSIZE
set MOVE [expr 10-10*$PSLABELSIZE]
foreach name {plan hopf elev} {
.$name lower PS${name}1
global FONT1
torus_pieces $name 1
.${name} scale PS${name}1 0 0 $PSLABELSIZE $PSLABELSIZE
.${name} move PS${name}1 0 ${MOVE}c 
torus_pieces $name 2 
.${name} scale PS${name}2 0 0 $PSLABELSIZE $PSLABELSIZE 
.${name} move PS${name}2 ${MOVE}c ${MOVE}c
.${name} create text .3c .3c  -fill white \
-font $FONT1 -anchor w -text "" -tag PSG${name}1 
.${name} create text 9.7c .3c -fill white \
-font $FONT1 -anchor e -text "" -tag PSG${name}2
.$name lower PS${name}1
.$name lower PSG${name}1
.$name lower PS${name}2
.$name lower PSG${name}2}
}

proc ps_memory {s k} {
global XGG PSBACK PSLABEL PSLABELSIZE
if {$PSLABEL == "yes"} {
.$s raise ps
.$s raise PS${s}${k}
.$s raise PSG${s}${k}}
for {set i 0} {$i<10} {incr i} {
for {set j 0} {$j<10} {incr j} {
.$s itemconfigure X${i}${j}X$k -fill $PSBACK
.$s itemconfigure ${i}y${j}$k -fill #888}}
set memory [.torus gettags back]
for {set i 0} {$i<10} {incr i} {
for {set j 0} {$j<10} {incr j} {
set tag X${k}${s}X${i}${j}X
set match1 [lsearch $memory $tag]
set match2 [lsearch $memory off$tag]
if {($match1 != -1) && ($match2 == -1)} {
set color [lindex $memory [expr [lsearch $memory ${tag}C*]]] 
set len [string length $color]
set color [string range $color [expr $len - 6] [expr $len -1]]
.$s itemconfigure X${i}${j}X$k -fill #$color
.$s itemconfigure PSG${s}$k -text $XGG(${k}${s}) -fill #888
}}}
}


proc ps_back {s} {
global PSBACK COL COL_COUNT PSLINES
if {$s==0} {set PSBACK #$COL($COL_COUNT)}
if {$s==1} {set PSLINES #$COL($COL_COUNT)}
partial_clear_torus
}

proc ps_label {} {
global GG_count OPTION PSLABEL PSBACK
if {$PSLABEL == "yes"} {
set pstemp "no"
foreach name {hopf plan elev} {
.$name lower PS${name}1
.$name lower PS${name}2}
}
if {$PSLABEL == "no"} {
set pstemp "yes"
if {$OPTION == 1} {.elev raise PSelev1}
if {$OPTION != 1} {
foreach s {hopf plan elev} {
for {set k 1} {$k  <= $GG_count($s)} {incr k} {
ps_memory $s $k
.$s raise PS${s}$k
}}}}
set PSLABEL $pstemp 
if {$OPTION == 1} {label_text}
}

proc ps_size {n} {
global PSSIZE OPTION
set pstemp $PSSIZE
if {($n == 0) & ($PSSIZE>.24)} {
set pstemp [expr $PSSIZE*.909090]
.torus scale pssize 9.15c 6.8c .909090 .909090} 
if {($n == 1) & ($PSSIZE<.995)} {
set pstemp [expr $PSSIZE*1.1]
.torus scale pssize 9.15c 6.8c 1.1 1.1} 
set PSSIZE $pstemp
if {$OPTION == 1} {pssize_text}
}

proc ps_labelsize {n} {
global PSLABELSIZE OPTION
set pstemp $PSLABELSIZE
if {($n == 0) & ($PSLABELSIZE>.15)} {
set pstemp [expr $PSLABELSIZE*.909090]
.hopf scale PShopf1 0c 10c .909090 .909090 
.plan scale PSplan1 0c 10c .909090 .909090 
.elev scale PSelev1 0c 10c .909090 .909090
.hopf scale PShopf2 10c 10c .909090 .909090 
.plan scale PSplan2 10c 10c .909090 .909090 
.elev scale PSelev2 10c 10c .909090 .909090} 

if {($n == 1) & ($PSLABELSIZE<.40)} {
set pstemp [expr $PSLABELSIZE*1.1]
.hopf scale PShopf1 0c 10c 1.1 1.1 
.plan scale PSplan1 0c 10c 1.1 1.1 
.elev scale PSelev1 0c 10c 1.1 1.1 
.hopf scale PShopf2 10c 10c 1.1 1.1 
.plan scale PSplan2 10c 10c 1.1 1.1 
.elev scale PSelev2 10c 10c 1.1 1.1} 

if {$OPTION == 1} {label_text}

set PSLABELSIZE $pstemp
}


#PARAMETERS
proc set_parameters {} {
global PROOF_ON PSLABEL PSBACK PSSIZE PSLABELSIZE 
global PARAMETER COORD FONT FONT0 FONT1 FONT2 FONT3 FONT4 OPTION
global COL_COUNT
set FONT0 "-*-times-medium-r-normal--*-350-*-*-*-*-*-*"
set FONT1 "-*-times-medium-r-normal--*-140-*-*-*-*-*-*"
set FONT2 "-*-times-medium-r-normal--*-120-*-*-*-*-*-*"
set FONT3 "-*-times-medium-r-normal--*-100-*-*-*-*-*-*"
set FONT4 "-*-times-medium-r-normal--*-180-*-*-*-*-*-*"
set FONT $FONT1
set OPTION 1
set COORD 3 
set PARAMETER 0
set PSBACK black
set PSSIZE 1.0
set PSLABELSIZE 0.25
set PSLABEL "no"
set PROOF_ON 0
set COL_COUNT 1X8
set PSLINES white
}

#################MAIN ROUTINES##########
proc main {} {
global PSLINES
set_parameters
catch {exec rm surface.o}
catch {exec gcc surface.c -lm}
catch {exec mv a.out surface.o}
board_layout
controls
torus_init
torus_pieces torus 1
ps_memory_setup
scale_init
width 1
height 1
many 0
parameter 0
option tutorial
set PSLINES white
clear_torus

}

main
